/**
 * StreamShield - Smart Popup Blocker for Streaming Sites
 * Specifically designed for sites like animekai, zoechip, fmovies, etc.
 *
 * This script provides targeted popup/redirect blocking for streaming sites
 * without breaking site functionality.
 */

(function() {
  'use strict';

  // Detect if we're on a streaming site
  const hostname = window.location.hostname.toLowerCase();
  const streamingSites = [
    'animekai', 'anikai', '9anime', 'zoro', 'hianime', 'aniwatch',
    'zoechip', 'lookmovie', 'fmovies', 'gomovies', 'putlocker',
    '123movies', 'soap2day', 'movie4k', 'flixmomo', 'rabbitstream',
    'vidstream', 'dokicloud', 'streamtape', 'mixdrop', 'yflix'
  ];

  // Sites that need ULTRA PASSIVE mode - only minimal intervention
  // These sites have aggressive anti-bot protection OR use redirects for video playback
  const ultraPassiveSites = [
    'animekai', 'anikai', 'hianime', 'megacloud', 'rapid-cloud', 'yflix',
    'zoechip', 'fmovies', 'gomovies', 'lookmovie', '123movies', 'soap2day',
    'putlocker', 'movie4k', 'flixmomo'
  ];

  const isStreamingSite = streamingSites.some(site => hostname.includes(site));
  const isUltraPassive = ultraPassiveSites.some(site => hostname.includes(site));

  if (!isStreamingSite) {
    console.log('StreamShield Popup Blocker: Not a streaming site, skipping');
    return;
  }

  // For ultra-passive sites like AnimeKai, skip ALL blocking to avoid rate limits
  if (isUltraPassive) {
    console.log('🛡️ StreamShield: Ultra-passive mode for', hostname, '(avoiding rate limits)');
    console.log('✅ Video playback prioritized over popup blocking');
    return; // Exit completely - no intervals, no observers
  }

  console.log('🛡️ StreamShield: Smart Popup/Redirect Blocker Active for', hostname);

  // ========================================================================
  // STATE TRACKING
  // ========================================================================
  let blockedPopups = 0;
  let clickTime = 0;
  let isUserClick = false;
  const originalLocation = window.location.href;
  let locationCheckCount = 0;

  // ========================================================================
  // BLOCK ALL POPUPS - NO EXCEPTIONS
  // ========================================================================
  const originalOpen = window.open;
  window.open = function(url, name, features) {
    console.log('🚫 BLOCKED popup:', url);
    blockedPopups++;
    return null;
  };

  // ========================================================================
  // TRACK USER CLICKS
  // ========================================================================
  document.addEventListener('click', function(e) {
    isUserClick = true;
    clickTime = Date.now();
    setTimeout(() => { isUserClick = false; }, 100);

    // Check if click target is an ad link
    const target = e.target.closest('a');
    if (target) {
      const href = target.href || '';
      if (isAdUrl(href)) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        console.log('🚫 BLOCKED ad link click:', href);
        blockedPopups++;
        return false;
      }
    }
  }, true);

  // ========================================================================
  // PREVENT REDIRECT LOOPS
  // ========================================================================
  const checkLocation = setInterval(() => {
    if (window.location.href !== originalLocation) {
      locationCheckCount++;
      if (locationCheckCount > 3 && !isUserClick) {
        console.log('🚫 BLOCKED redirect loop');
        window.location.href = originalLocation;
        clearInterval(checkLocation);
      }
    }
  }, 100);

  // ========================================================================
  // BLOCK FORCED REDIRECTS
  // ========================================================================
  const originalReplace = window.location.replace;
  const originalAssign = window.location.assign;

  window.location.replace = function(url) {
    if (!isUserClick && url !== window.location.href) {
      console.log('🚫 BLOCKED location.replace:', url);
      return;
    }
    return originalReplace.call(window.location, url);
  };

  window.location.assign = function(url) {
    if (!isUserClick && url !== window.location.href) {
      console.log('🚫 BLOCKED location.assign:', url);
      return;
    }
    return originalAssign.call(window.location, url);
  };

  // ========================================================================
  // BLOCK TIMER-BASED POPUPS
  // ========================================================================
  const originalSetTimeout = window.setTimeout;
  const originalSetInterval = window.setInterval;

  window.setTimeout = function(fn, delay) {
    if (typeof fn === 'string' && (fn.includes('window.open') || fn.includes('location'))) {
      console.log('🚫 BLOCKED setTimeout redirect');
      return 0;
    }
    return originalSetTimeout.apply(this, arguments);
  };

  window.setInterval = function(fn, delay) {
    if (typeof fn === 'string' && (fn.includes('window.open') || fn.includes('location'))) {
      console.log('🚫 BLOCKED setInterval redirect');
      return 0;
    }
    return originalSetInterval.apply(this, arguments);
  };

  // ========================================================================
  // AD URL DETECTION
  // ========================================================================
  function isAdUrl(url) {
    if (!url) return false;
    const lowUrl = url.toLowerCase();

    const adPatterns = [
      '/rei.php', '/go.php', '/out.php', '/click.php', '/redir',
      'popads', 'popunder', 'popups', 'clickunder',
      'adserver', 'adsterra', 'exoclick', 'juicyads',
      'propellerads', 'admaven', 'clickadu', 'trafficjunky',
      'adsblocked.app', 'adblocked.app', 'highperformance',
      'bidvertiser', 'onclickads', 'onclickmax', 'richpush',
      'notix.io', 'push-ad', 'pushprofit', 'webpusher',
      'betterads.co', 'streamads.net', 'clicksgear.com',
      'adxpremium', 'hrtya.com', 'hrtyb.com', 'hrtye.com',
      'ladsads.com', 'newpopunder.com', 'popmyads.com',
      'revenuehits.com', 'acint.net', 'trafficforce.com',
      'landingtrackone.com', 'popunderjs.com', 'popupads.ir',
      // AnimeKai-specific popup domains (from uBlock Origin issues #29438, #29557)
      '1movies.bz', '1movies.', 'onclickperformance.com',
      'onclasrv.com', 'oclasrv.com', 'exdynsrv.com',
      'streamtapeadblock.art', 'popcash.net', 'hilltopads.net',
      // Common anime streaming popup patterns
      'a-ads.com', 'ad-maven.com', 'ad4mat.com', 'adtng.com',
      'realsrv.com', 'tsyndicate.com', 'cpmstar.com',
      // ZoeChip-specific ad domains
      'woovoree.net', 'woovoree.com'
    ];

    return adPatterns.some(pattern => lowUrl.includes(pattern));
  }

  // ========================================================================
  // SMART AD OVERLAY REMOVAL - Targeted, not aggressive
  // ========================================================================

  // Word boundary patterns for accurate ad detection (avoids matching "loaded", "header", etc.)
  const adWordPatterns = [
    /\bad\b/i, /\bads\b/i, /\badvert/i, /\badvertisement/i,
    /\bpopup\b/i, /\bpop-up\b/i, /\bpopunder\b/i, /\bpop-under\b/i,
    /\boverlay-ad\b/i, /\bad-overlay\b/i
  ];

  // Known ad network patterns in iframe sources
  const adIframeSources = [
    'doubleclick', 'googlesyndication', 'googleadservices',
    'adsterra', 'exoclick', 'juicyads', 'propellerads',
    'popads', 'popunder', 'trafficjunky', 'bidvertiser'
  ];

  // Elements to NEVER remove (site functionality)
  const protectedSelectors = [
    'video', 'nav', 'header', 'footer', 'main', 'article',
    '[class*="player"]', '[id*="player"]',
    '[class*="video"]', '[id*="video"]',
    '[class*="episode"]', '[id*="episode"]',
    '[class*="anime"]', '[class*="movie"]', '[class*="show"]',
    '[class*="search"]', '[class*="menu"]', '[class*="nav"]',
    '[class*="sidebar"]', '[class*="content"]', '[class*="container"]'
  ];

  function isProtectedElement(el) {
    // Check if element matches any protected selector
    for (const selector of protectedSelectors) {
      try {
        if (el.matches(selector)) return true;
        // Also check if it contains protected elements
        if (el.querySelector(selector)) return true;
      } catch(e) {}
    }
    return false;
  }

  function isAdElement(el) {
    const id = (el.id || '').toLowerCase();
    const className = (el.className || '').toString().toLowerCase();
    const combined = id + ' ' + className;

    // Check word boundary patterns
    for (const pattern of adWordPatterns) {
      if (pattern.test(combined)) return true;
    }

    // Check for ad iframes
    if (el.tagName === 'IFRAME') {
      const src = (el.src || '').toLowerCase();
      for (const adSrc of adIframeSources) {
        if (src.includes(adSrc)) return true;
      }
    }

    return false;
  }

  function removeAdOverlays() {
    // Only target specific ad patterns, not all fixed/absolute elements
    const suspiciousElements = document.querySelectorAll(`
      [id*="advert"], [class*="advert"],
      [id*="ad-slot"], [class*="ad-slot"],
      [id*="ad_slot"], [class*="ad_slot"],
      [id*="adcontainer"], [class*="adcontainer"],
      [id*="ad-container"], [class*="ad-container"],
      [id*="ad_container"], [class*="ad_container"],
      [id*="popunder"], [class*="popunder"],
      [id*="popup-ad"], [class*="popup-ad"],
      iframe[src*="doubleclick"],
      iframe[src*="googlesyndication"],
      iframe[src*="adsterra"],
      iframe[src*="exoclick"],
      iframe[src*="popads"]
    `);

    suspiciousElements.forEach(el => {
      // Double-check it's not protected
      if (isProtectedElement(el)) return;

      // Double-check it's actually an ad
      if (isAdElement(el)) {
        console.log('🗑️ Removed ad element:', {
          tag: el.tagName,
          id: el.id,
          class: (el.className || '').toString().substring(0, 30)
        });
        el.remove();
      }
    });

    // Re-enable scrolling if disabled by ad scripts
    if (document.body.style.overflow === 'hidden') {
      // Only restore if there's no legitimate modal open
      const hasLegitModal = document.querySelector('.modal.show, [role="dialog"]:not([aria-hidden="true"])');
      if (!hasLegitModal) {
        document.body.style.overflow = '';
        document.documentElement.style.overflow = '';
      }
    }
  }

  // Run overlay removal less frequently to avoid performance issues
  removeAdOverlays();
  setInterval(removeAdOverlays, 2000);  // Every 2 seconds (not 500ms)

  // ========================================================================
  // REMOVE RANDOM-NAMED CLICK INTERCEPT OVERLAYS (ZoeChip specific)
  // These are empty divs with random class names that intercept all clicks
  // ========================================================================
  function removeClickInterceptOverlays() {
    let removed = 0;
    document.querySelectorAll('div').forEach(el => {
      // Skip if element has children (not an overlay)
      if (el.children.length > 0) return;

      // Skip if no class or class contains common legitimate patterns
      const className = el.className || '';
      if (!className || className.length < 5) return;
      if (className.includes('-') || className.includes('_') || className.includes(' ')) return;

      // Check if it's a high z-index fixed/absolute positioned overlay
      const style = window.getComputedStyle(el);
      const zIndex = parseInt(style.zIndex) || 0;
      const position = style.position;

      if ((position === 'fixed' || position === 'absolute') && zIndex > 100) {
        // Check if it covers significant screen area (click intercept overlays are usually fullscreen)
        const rect = el.getBoundingClientRect();
        const screenCoverage = (rect.width * rect.height) / (window.innerWidth * window.innerHeight);

        if (screenCoverage > 0.3) {  // Covers more than 30% of screen
          console.log('🗑️ Removed click intercept overlay:', className);
          el.remove();
          removed++;
        }
      }
    });
    return removed;
  }

  // Run click intercept removal frequently (these regenerate)
  removeClickInterceptOverlays();
  setInterval(removeClickInterceptOverlays, 500);  // Every 500ms

  // ========================================================================
  // REMOVE ONCLICK ATTRIBUTES
  // ========================================================================
  function removeOnClickHandlers() {
    let removed = 0;
    document.querySelectorAll('[onclick]').forEach(el => {
      const onclick = el.getAttribute('onclick') || '';
      if (onclick.includes('window.open') ||
          onclick.includes('location.href') ||
          onclick.includes('location.replace')) {
        el.removeAttribute('onclick');
        removed++;
      }
    });
    if (removed > 0) {
      console.log(`🚫 Removed ${removed} suspicious onclick handlers`);
    }
  }

  setInterval(removeOnClickHandlers, 1000);

  // ========================================================================
  // MONITOR FOR NEW IFRAMES - Only block known ad network iframes
  // ========================================================================
  const observer = new MutationObserver(function(mutations) {
    mutations.forEach(function(mutation) {
      mutation.addedNodes.forEach(function(node) {
        if (node.tagName === 'IFRAME') {
          const src = (node.src || '').toLowerCase();

          // Only block iframes from known ad networks
          const isAdIframe = adIframeSources.some(adSrc => src.includes(adSrc));

          // Also check for suspicious patterns in URL
          const suspiciousPatterns = [
            '/ad.', '/ads/', '/advert', '/banner',
            'popunder', 'popads', 'clickunder'
          ];
          const hasSuspiciousPattern = suspiciousPatterns.some(pattern => src.includes(pattern));

          // Block if it's from an ad network OR has suspicious URL pattern
          if (isAdIframe || hasSuspiciousPattern) {
            console.log('🚫 BLOCKED ad iframe:', src.substring(0, 80));
            node.remove();
          }

          // Note: We do NOT block iframes just because their ID/class contains "ad"
          // This was causing false positives on streaming sites
        }
      });
    });
  });

  observer.observe(document.documentElement, {
    childList: true,
    subtree: true
  });

  // ========================================================================
  // PREVENT DOCUMENT EVENT HANDLERS
  // ========================================================================
  try {
    Object.defineProperty(document, 'onclick', { value: null, writable: false });
    Object.defineProperty(document, 'onmousedown', { value: null, writable: false });
  } catch(e) {
    // Some browsers don't allow this
  }

  // ========================================================================
  // BLOCK CONTEXT MENU POPUPS
  // ========================================================================
  document.addEventListener('contextmenu', function(e) {
    // Allow context menu, but prevent popup scripts
    const link = e.target.closest('a');
    if (link && isAdUrl(link.href)) {
      e.preventDefault();
      e.stopImmediatePropagation();
      return false;
    }
  }, true);

  // ========================================================================
  // REPORT STATS
  // ========================================================================
  setInterval(() => {
    if (blockedPopups > 0) {
      console.log(`📊 StreamShield: Blocked ${blockedPopups} popups/redirects`);
    }
  }, 30000); // Report every 30 seconds

  console.log('✅ Free streaming popup/redirect blocker loaded');

})();
