/**
 * StreamShield SponsorBlock Integration
 *
 * Content script for YouTube that integrates with the SponsorBlock API
 * to automatically skip sponsored segments, intros, outros, and other
 * non-content portions of videos.
 *
 * PREMIUM FEATURE: SponsorBlock integration is a premium-only feature.
 *
 * @author StreamShield Team
 * @version 1.1.0
 */

(function() {
    'use strict';

    // =========================================================================
    // LOGGING UTILITIES
    // =========================================================================

    function log(...args) {
        console.log('[StreamShield SponsorBlock]', ...args);
    }

    function warn(...args) {
        console.warn('[StreamShield SponsorBlock]', ...args);
    }

    function error(...args) {
        console.error('[StreamShield SponsorBlock]', ...args);
    }

    function debug(...args) {
        if (CONFIG.DEBUG) {
            console.debug('[StreamShield SponsorBlock]', ...args);
        }
    }

    // =========================================================================
    // CONFIGURATION
    // =========================================================================

    const CONFIG = {
        API_BASE: 'https://sponsor.ajay.app/api',
        CACHE_DURATION: 3600000, // 1 hour in milliseconds
        SKIP_NOTICE_DURATION: 3000, // 3 seconds
        SEGMENT_UPDATE_INTERVAL: 250, // Check video time every 250ms
        DEBUG: true, // Enable debug logging

        // Segment categories and their colors (matching SponsorBlock)
        CATEGORIES: {
            sponsor: { color: '#00D400', label: 'Sponsor', default: true },
            intro: { color: '#00FFFF', label: 'Intro', default: true },
            outro: { color: '#0202ED', label: 'Outro', default: true },
            interaction: { color: '#CC00FF', label: 'Interaction Reminder', default: true },
            selfpromo: { color: '#FFFF00', label: 'Self Promotion', default: false },
            music_offtopic: { color: '#FF9900', label: 'Off-topic Music', default: false },
            preview: { color: '#008FD6', label: 'Preview/Recap', default: false },
            filler: { color: '#7300FF', label: 'Filler', default: false }
        }
    };

    // =========================================================================
    // STATE
    // =========================================================================

    const state = {
        videoId: null,
        segments: [],
        video: null,
        progressBar: null,
        markers: [],
        skipNotice: null,
        timeUpdateHandler: null,
        lastSkippedSegment: null,
        categorySettings: {},
        initialized: false,
        observing: false,
        isPremium: false,
        premiumChecked: false
    };

    // =========================================================================
    // PREMIUM CHECK
    // =========================================================================

    /**
     * Check if user has premium access for SponsorBlock
     * @returns {Promise<boolean>}
     */
    async function checkPremiumAccess() {
        if (state.premiumChecked) {
            return state.isPremium;
        }

        return new Promise((resolve) => {
            if (typeof chrome === 'undefined' || !chrome.storage) {
                log('Chrome storage not available, disabling SponsorBlock');
                state.premiumChecked = true;
                state.isPremium = false;
                resolve(false);
                return;
            }

            try {
                chrome.storage.sync.get(['streamshieldLicense'], (result) => {
                    if (chrome.runtime.lastError) {
                        warn('Failed to check premium status:', chrome.runtime.lastError.message);
                        state.premiumChecked = true;
                        state.isPremium = false;
                        resolve(false);
                        return;
                    }

                    if (result.streamshieldLicense) {
                        const license = result.streamshieldLicense;
                        // Check if still valid
                        if (license.lifetime) {
                            log('Lifetime premium license active');
                            state.isPremium = true;
                        } else if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
                            log('Premium license active, expires:', license.expiresAt);
                            state.isPremium = true;
                        } else {
                            debug('Premium license expired');
                            state.isPremium = false;
                        }
                    } else {
                        debug('No premium license found');
                        state.isPremium = false;
                    }

                    state.premiumChecked = true;
                    resolve(state.isPremium);
                });
            } catch (e) {
                error('Exception checking premium status:', e.message);
                state.premiumChecked = true;
                state.isPremium = false;
                resolve(false);
            }
        });
    }

    /**
     * Show premium required notice in YouTube player
     */
    function showPremiumRequiredNotice() {
        // Find video container
        const container = document.querySelector('.html5-video-container') ||
                         document.querySelector('#movie_player') ||
                         document.querySelector('.html5-video-player');

        if (!container) {
            debug('Could not find video container for premium notice');
            return;
        }

        // Check if notice already exists
        if (document.getElementById('streamshield-sb-premium-notice')) {
            return;
        }

        const notice = document.createElement('div');
        notice.id = 'streamshield-sb-premium-notice';
        notice.style.cssText = [
            'position: absolute',
            'bottom: 80px',
            'left: 50%',
            'transform: translateX(-50%)',
            'background: linear-gradient(135deg, rgba(26, 26, 46, 0.95), rgba(22, 33, 62, 0.95))',
            'color: white',
            'padding: 12px 20px',
            'border-radius: 8px',
            'font-family: YouTube Noto, Roboto, Arial, sans-serif',
            'font-size: 13px',
            'z-index: 9999',
            'display: flex',
            'align-items: center',
            'gap: 12px',
            'box-shadow: 0 4px 20px rgba(0, 0, 0, 0.4)',
            'border: 1px solid rgba(255, 255, 255, 0.1)',
            'pointer-events: auto'
        ].join('; ');

        const icon = document.createElement('span');
        icon.textContent = '\uD83D\uDD12'; // Lock emoji
        icon.style.fontSize = '18px';

        const text = document.createElement('span');
        text.textContent = 'SponsorBlock is a Premium feature';

        const upgradeBtn = document.createElement('button');
        upgradeBtn.textContent = 'Upgrade';
        upgradeBtn.style.cssText = [
            'background: linear-gradient(135deg, #00C853, #00E676)',
            'color: #000',
            'border: none',
            'padding: 6px 12px',
            'border-radius: 4px',
            'font-size: 12px',
            'font-weight: 600',
            'cursor: pointer',
            'transition: transform 0.2s'
        ].join('; ');
        upgradeBtn.onmouseover = function() { upgradeBtn.style.transform = 'scale(1.05)'; };
        upgradeBtn.onmouseout = function() { upgradeBtn.style.transform = 'scale(1)'; };
        upgradeBtn.onclick = function() {
            window.open('https://streamshield.io/premium', '_blank');
        };

        const closeBtn = document.createElement('button');
        closeBtn.textContent = '\u2715';
        closeBtn.style.cssText = [
            'background: transparent',
            'color: #888',
            'border: none',
            'padding: 4px 8px',
            'font-size: 14px',
            'cursor: pointer',
            'margin-left: 4px'
        ].join('; ');
        closeBtn.onclick = function() { notice.remove(); };

        notice.appendChild(icon);
        notice.appendChild(text);
        notice.appendChild(upgradeBtn);
        notice.appendChild(closeBtn);
        container.appendChild(notice);

        // Auto-remove after 8 seconds
        setTimeout(function() {
            if (notice.parentNode) {
                notice.style.opacity = '0';
                notice.style.transition = 'opacity 0.3s';
                setTimeout(function() { notice.remove(); }, 300);
            }
        }, 8000);

        log('Premium required notice displayed');
    }

    // =========================================================================
    // UTILITY FUNCTIONS
    // =========================================================================

    /**
     * Extract video ID from YouTube URL
     * Handles various YouTube URL formats
     */
    function extractVideoId(url) {
        const urlObj = new URL(url);

        // Standard watch URL: youtube.com/watch?v=VIDEO_ID
        if (urlObj.pathname === '/watch') {
            return urlObj.searchParams.get('v');
        }

        // Shorts: youtube.com/shorts/VIDEO_ID
        const shortsMatch = urlObj.pathname.match(/^\/shorts\/([a-zA-Z0-9_-]{11})/);
        if (shortsMatch) {
            return shortsMatch[1];
        }

        // Embed: youtube.com/embed/VIDEO_ID
        const embedMatch = urlObj.pathname.match(/^\/embed\/([a-zA-Z0-9_-]{11})/);
        if (embedMatch) {
            return embedMatch[1];
        }

        // Live: youtube.com/live/VIDEO_ID
        const liveMatch = urlObj.pathname.match(/^\/live\/([a-zA-Z0-9_-]{11})/);
        if (liveMatch) {
            return liveMatch[1];
        }

        return null;
    }

    /**
     * Check if current page is a YouTube video page
     */
    function isVideoPage() {
        return window.location.hostname.includes('youtube.com') &&
               extractVideoId(window.location.href) !== null;
    }

    /**
     * Format time in seconds to MM:SS or HH:MM:SS
     */
    function formatTime(seconds) {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = Math.floor(seconds % 60);

        if (hours > 0) {
            return hours + ':' + minutes.toString().padStart(2, '0') + ':' + secs.toString().padStart(2, '0');
        }
        return minutes + ':' + secs.toString().padStart(2, '0');
    }

    /**
     * Escape text for safe display
     */
    function escapeText(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.textContent;
    }

    // =========================================================================
    // STORAGE MANAGEMENT
    // =========================================================================

    /**
     * Load category settings from chrome.storage
     */
    async function loadCategorySettings() {
        return new Promise((resolve) => {
            if (typeof chrome === 'undefined' || !chrome.storage) {
                // Default settings if storage unavailable
                Object.keys(CONFIG.CATEGORIES).forEach(cat => {
                    state.categorySettings[cat] = CONFIG.CATEGORIES[cat].default;
                });
                resolve(state.categorySettings);
                return;
            }

            chrome.storage.sync.get(['sponsorblock_categories'], (result) => {
                if (chrome.runtime.lastError) {
                    log('Storage error:', chrome.runtime.lastError);
                }

                if (result.sponsorblock_categories) {
                    state.categorySettings = result.sponsorblock_categories;
                } else {
                    // Initialize with defaults
                    Object.keys(CONFIG.CATEGORIES).forEach(cat => {
                        state.categorySettings[cat] = CONFIG.CATEGORIES[cat].default;
                    });
                }

                log('Loaded category settings:', state.categorySettings);
                resolve(state.categorySettings);
            });
        });
    }

    /**
     * Get enabled categories for API request
     */
    function getEnabledCategories() {
        return Object.keys(state.categorySettings)
            .filter(cat => state.categorySettings[cat]);
    }

    /**
     * Cache segments in sessionStorage
     */
    function cacheSegments(videoId, segments) {
        try {
            const cacheData = {
                segments,
                timestamp: Date.now()
            };
            sessionStorage.setItem('sb_cache_' + videoId, JSON.stringify(cacheData));
        } catch (e) {
            log('Cache write error:', e);
        }
    }

    /**
     * Get cached segments if still valid
     */
    function getCachedSegments(videoId) {
        try {
            const cached = sessionStorage.getItem('sb_cache_' + videoId);
            if (cached) {
                const cacheData = JSON.parse(cached);
                if (Date.now() - cacheData.timestamp < CONFIG.CACHE_DURATION) {
                    return cacheData.segments;
                }
            }
        } catch (e) {
            log('Cache read error:', e);
        }
        return null;
    }

    // =========================================================================
    // API INTERACTION
    // =========================================================================

    /**
     * Fetch segments from SponsorBlock API
     */
    async function fetchSegments(videoId) {
        // Check cache first
        const cached = getCachedSegments(videoId);
        if (cached) {
            log('Using cached segments for', videoId);
            return cached;
        }

        const enabledCategories = getEnabledCategories();
        if (enabledCategories.length === 0) {
            log('No categories enabled');
            return [];
        }

        const categoriesParam = JSON.stringify(enabledCategories);
        const url = CONFIG.API_BASE + '/skipSegments?videoID=' + videoId + '&categories=' + encodeURIComponent(categoriesParam);

        try {
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json'
                }
            });

            if (response.status === 404) {
                // No segments found for this video
                log('No segments found for', videoId);
                cacheSegments(videoId, []);
                return [];
            }

            if (!response.ok) {
                throw new Error('API error: ' + response.status);
            }

            const segments = await response.json();
            log('Fetched segments:', segments);

            // Cache the results
            cacheSegments(videoId, segments);

            return segments;
        } catch (err) {
            error('Failed to fetch segments:', err);
            return [];
        }
    }

    // =========================================================================
    // SKIP NOTIFICATION UI
    // =========================================================================

    /**
     * Create the skip notification element
     */
    function createSkipNotice() {
        if (state.skipNotice) return state.skipNotice;

        const notice = document.createElement('div');
        notice.id = 'streamshield-skip-notice';
        notice.style.cssText = [
            'position: absolute',
            'bottom: 80px',
            'left: 50%',
            'transform: translateX(-50%)',
            'background: rgba(0, 0, 0, 0.85)',
            'color: white',
            'padding: 10px 20px',
            'border-radius: 8px',
            'font-family: YouTube Noto, Roboto, Arial, sans-serif',
            'font-size: 14px',
            'z-index: 9999',
            'display: none',
            'align-items: center',
            'gap: 10px',
            'box-shadow: 0 4px 12px rgba(0, 0, 0, 0.4)',
            'transition: opacity 0.3s ease',
            'pointer-events: auto'
        ].join('; ');

        const icon = document.createElement('span');
        icon.textContent = '\u23ED'; // Fast-forward icon
        icon.style.fontSize = '16px';

        const text = document.createElement('span');
        text.className = 'skip-text';

        const undoButton = document.createElement('button');
        undoButton.textContent = 'Undo';
        undoButton.style.cssText = [
            'background: rgba(255, 255, 255, 0.2)',
            'border: none',
            'color: white',
            'padding: 5px 12px',
            'border-radius: 4px',
            'cursor: pointer',
            'font-size: 12px',
            'margin-left: 8px',
            'transition: background 0.2s'
        ].join('; ');
        undoButton.onmouseover = function() { undoButton.style.background = 'rgba(255, 255, 255, 0.3)'; };
        undoButton.onmouseout = function() { undoButton.style.background = 'rgba(255, 255, 255, 0.2)'; };
        undoButton.onclick = handleUndo;

        notice.appendChild(icon);
        notice.appendChild(text);
        notice.appendChild(undoButton);

        state.skipNotice = notice;
        return notice;
    }

    /**
     * Show skip notification
     */
    function showSkipNotice(category, startTime, endTime) {
        const notice = state.skipNotice || createSkipNotice();
        const categoryInfo = CONFIG.CATEGORIES[category] || { label: category, color: '#888' };

        const text = notice.querySelector('.skip-text');
        // Clear existing content
        while (text.firstChild) {
            text.removeChild(text.firstChild);
        }

        // Build content safely using DOM methods
        const skippedText = document.createTextNode('Skipped ');
        const labelSpan = document.createElement('span');
        labelSpan.style.color = categoryInfo.color;
        labelSpan.style.fontWeight = 'bold';
        labelSpan.textContent = escapeText(categoryInfo.label);

        text.appendChild(skippedText);
        text.appendChild(labelSpan);

        // Attach to video container
        const container = document.querySelector('.html5-video-container') ||
                         document.querySelector('#movie_player') ||
                         document.querySelector('.html5-video-player');

        if (container && !notice.parentElement) {
            container.appendChild(notice);
        }

        notice.style.display = 'flex';
        notice.style.opacity = '1';

        // Store skip info for undo
        state.lastSkippedSegment = {
            category: category,
            startTime: startTime,
            endTime: endTime,
            skippedAt: Date.now()
        };

        // Auto-hide after duration
        clearTimeout(notice._hideTimeout);
        notice._hideTimeout = setTimeout(function() {
            notice.style.opacity = '0';
            setTimeout(function() {
                notice.style.display = 'none';
            }, 300);
        }, CONFIG.SKIP_NOTICE_DURATION);
    }

    /**
     * Handle undo button click
     */
    function handleUndo() {
        if (!state.lastSkippedSegment || !state.video) return;

        var startTime = state.lastSkippedSegment.startTime;
        state.video.currentTime = startTime;
        state.lastSkippedSegment = null;

        // Hide notice immediately
        if (state.skipNotice) {
            state.skipNotice.style.display = 'none';
        }

        log('Undid skip, returned to', startTime);
    }

    // =========================================================================
    // PROGRESS BAR MARKERS
    // =========================================================================

    /**
     * Find YouTube's progress bar element
     */
    function findProgressBar() {
        // Try multiple selectors for different YouTube layouts
        return document.querySelector('.ytp-progress-bar-container') ||
               document.querySelector('.ytp-progress-bar') ||
               document.querySelector('.ytp-timed-markers-container');
    }

    /**
     * Create segment markers on the progress bar
     */
    function createProgressBarMarkers() {
        const progressBar = findProgressBar();
        if (!progressBar || !state.video) {
            log('Progress bar or video not found');
            return;
        }

        // Remove existing markers
        removeProgressBarMarkers();

        const duration = state.video.duration;
        if (!duration || duration <= 0) {
            log('Video duration not available');
            return;
        }

        // Create marker container
        const markerContainer = document.createElement('div');
        markerContainer.id = 'streamshield-segment-markers';
        markerContainer.style.cssText = [
            'position: absolute',
            'top: 0',
            'left: 0',
            'right: 0',
            'bottom: 0',
            'pointer-events: none',
            'z-index: 40'
        ].join('; ');

        state.segments.forEach(function(segment, index) {
            if (!state.categorySettings[segment.category]) return;

            const startPercent = (segment.segment[0] / duration) * 100;
            const endPercent = (segment.segment[1] / duration) * 100;
            const widthPercent = endPercent - startPercent;

            const marker = document.createElement('div');
            marker.className = 'streamshield-segment-marker';
            marker.dataset.category = segment.category;
            marker.dataset.index = index;

            const categoryInfo = CONFIG.CATEGORIES[segment.category] || { color: '#888' };

            marker.style.cssText = [
                'position: absolute',
                'left: ' + startPercent + '%',
                'width: ' + Math.max(widthPercent, 0.5) + '%',
                'height: 100%',
                'background-color: ' + categoryInfo.color,
                'opacity: 0.7',
                'pointer-events: none',
                'border-radius: 1px'
            ].join('; ');

            markerContainer.appendChild(marker);
            state.markers.push(marker);
        });

        // Insert markers into progress bar
        const scrubberContainer = progressBar.querySelector('.ytp-progress-list') || progressBar;
        scrubberContainer.style.position = 'relative';
        scrubberContainer.appendChild(markerContainer);

        log('Created', state.markers.length, 'progress bar markers');
    }

    /**
     * Remove all segment markers
     */
    function removeProgressBarMarkers() {
        const existingContainer = document.getElementById('streamshield-segment-markers');
        if (existingContainer) {
            existingContainer.remove();
        }
        state.markers = [];
    }

    // =========================================================================
    // SEGMENT SKIPPING
    // =========================================================================

    /**
     * Check current time and skip if in a segment
     */
    function checkAndSkip() {
        if (!state.video || state.segments.length === 0) return;

        const currentTime = state.video.currentTime;
        const skipBuffer = 0.5; // Small buffer to prevent re-skipping

        for (var i = 0; i < state.segments.length; i++) {
            var segment = state.segments[i];

            // Skip if category is disabled
            if (!state.categorySettings[segment.category]) continue;

            const startTime = segment.segment[0];
            const endTime = segment.segment[1];

            // Check if we're within the segment
            if (currentTime >= startTime && currentTime < endTime - skipBuffer) {
                // Don't skip the same segment twice in quick succession
                if (state.lastSkippedSegment &&
                    state.lastSkippedSegment.startTime === startTime &&
                    Date.now() - state.lastSkippedSegment.skippedAt < 2000) {
                    continue;
                }

                log('Skipping ' + segment.category + ' segment: ' + formatTime(startTime) + ' - ' + formatTime(endTime));
                state.video.currentTime = endTime;
                showSkipNotice(segment.category, startTime, endTime);
                break;
            }
        }
    }

    /**
     * Start monitoring video time for segment skipping
     */
    function startTimeMonitoring() {
        if (state.timeUpdateHandler) {
            clearInterval(state.timeUpdateHandler);
        }

        // Use interval instead of timeupdate event for more consistent behavior
        state.timeUpdateHandler = setInterval(checkAndSkip, CONFIG.SEGMENT_UPDATE_INTERVAL);
        log('Started time monitoring');
    }

    /**
     * Stop time monitoring
     */
    function stopTimeMonitoring() {
        if (state.timeUpdateHandler) {
            clearInterval(state.timeUpdateHandler);
            state.timeUpdateHandler = null;
        }
    }

    // =========================================================================
    // VIDEO ELEMENT HANDLING
    // =========================================================================

    /**
     * Find the main video element
     */
    function findVideoElement() {
        // Primary selector for YouTube's main video
        return document.querySelector('video.html5-main-video') ||
               document.querySelector('#movie_player video') ||
               document.querySelector('.html5-video-player video') ||
               document.querySelector('video');
    }

    /**
     * Set up video element handlers
     */
    function setupVideoHandlers() {
        const video = findVideoElement();
        if (!video) {
            log('Video element not found');
            return false;
        }

        state.video = video;

        // Handle video metadata loaded
        video.addEventListener('loadedmetadata', function() {
            log('Video metadata loaded, duration:', video.duration);
            createProgressBarMarkers();
        });

        // Handle seeking (recreate markers if needed)
        video.addEventListener('durationchange', function() {
            log('Duration changed:', video.duration);
            createProgressBarMarkers();
        });

        // Handle video play/pause
        video.addEventListener('play', startTimeMonitoring);
        video.addEventListener('pause', stopTimeMonitoring);

        // If video is already playing, start monitoring
        if (!video.paused) {
            startTimeMonitoring();
        }

        // Create markers if video is already loaded
        if (video.duration > 0) {
            createProgressBarMarkers();
        }

        return true;
    }

    // =========================================================================
    // PAGE CHANGE DETECTION (SPA HANDLING)
    // =========================================================================

    /**
     * Handle video/page changes
     */
    async function handleVideoChange() {
        const newVideoId = extractVideoId(window.location.href);

        if (newVideoId === state.videoId) {
            log('Same video, skipping refetch');
            return;
        }

        log('Video changed to:', newVideoId);

        // Clean up previous state
        cleanup();

        if (!newVideoId) {
            log('Not a video page');
            return;
        }

        state.videoId = newVideoId;

        // Load settings and fetch segments
        await loadCategorySettings();
        state.segments = await fetchSegments(newVideoId);

        if (state.segments.length > 0) {
            log('Found', state.segments.length, 'segments');

            // Wait for video element with retry
            var attempts = 0;
            const maxAttempts = 20;
            const setupInterval = setInterval(function() {
                attempts++;
                if (setupVideoHandlers() || attempts >= maxAttempts) {
                    clearInterval(setupInterval);
                    if (attempts >= maxAttempts) {
                        log('Failed to find video element after', maxAttempts, 'attempts');
                    }
                }
            }, 500);
        }
    }

    /**
     * Set up SPA navigation detection
     */
    function setupNavigationObserver() {
        if (state.observing) return;

        // YouTube uses History API for navigation
        const originalPushState = history.pushState;
        const originalReplaceState = history.replaceState;

        history.pushState = function() {
            originalPushState.apply(this, arguments);
            log('pushState detected');
            setTimeout(handleVideoChange, 100);
        };

        history.replaceState = function() {
            originalReplaceState.apply(this, arguments);
            log('replaceState detected');
            setTimeout(handleVideoChange, 100);
        };

        // Handle back/forward navigation
        window.addEventListener('popstate', function() {
            log('popstate detected');
            setTimeout(handleVideoChange, 100);
        });

        // Also watch for YouTube's specific navigation events
        window.addEventListener('yt-navigate-finish', function() {
            log('yt-navigate-finish detected');
            setTimeout(handleVideoChange, 100);
        });

        // MutationObserver for video element changes
        const observer = new MutationObserver(function(mutations) {
            for (var i = 0; i < mutations.length; i++) {
                var mutation = mutations[i];
                if (mutation.type === 'childList') {
                    var nodes = Array.from(mutation.addedNodes);
                    var addedVideo = nodes.find(function(node) {
                        return node.nodeName === 'VIDEO';
                    });
                    if (addedVideo) {
                        log('New video element detected');
                        setupVideoHandlers();
                        createProgressBarMarkers();
                    }
                }
            }
        });

        const playerContainer = document.getElementById('movie_player') ||
                               document.querySelector('.html5-video-player');
        if (playerContainer) {
            observer.observe(playerContainer, { childList: true, subtree: true });
        }

        state.observing = true;
        log('Navigation observer set up');
    }

    // =========================================================================
    // CLEANUP
    // =========================================================================

    /**
     * Clean up state and handlers
     */
    function cleanup() {
        stopTimeMonitoring();
        removeProgressBarMarkers();

        if (state.skipNotice) {
            state.skipNotice.remove();
            state.skipNotice = null;
        }

        state.segments = [];
        state.lastSkippedSegment = null;
        state.video = null;
    }

    // =========================================================================
    // EXTENSION COMMUNICATION
    // =========================================================================

    /**
     * Listen for messages from extension popup/background
     */
    function setupMessageListener() {
        if (typeof chrome === 'undefined' || !chrome.runtime) {
            warn('Chrome runtime not available for message listener');
            return;
        }

        try {
            chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
                try {
                    switch (request.action) {
                        case 'getSponsorBlockStatus':
                            sendResponse({
                                videoId: state.videoId,
                                segmentCount: state.segments.length,
                                categories: state.categorySettings,
                                isActive: state.video !== null,
                                isPremium: state.isPremium,
                                initialized: state.initialized
                            });
                            break;

                        case 'updateSponsorBlockCategories':
                            if (!state.isPremium) {
                                sendResponse({ success: false, error: 'Premium required' });
                                break;
                            }
                            state.categorySettings = request.categories;
                            // Refetch segments with new settings
                            if (state.videoId) {
                                fetchSegments(state.videoId).then(function(segments) {
                                    state.segments = segments;
                                    createProgressBarMarkers();
                                });
                            }
                            sendResponse({ success: true });
                            break;

                        case 'skipToSegmentEnd':
                            if (!state.isPremium) {
                                sendResponse({ success: false, error: 'Premium required' });
                                break;
                            }
                            if (state.video && request.segmentIndex < state.segments.length) {
                                var segment = state.segments[request.segmentIndex];
                                state.video.currentTime = segment.segment[1];
                                sendResponse({ success: true });
                            } else {
                                sendResponse({ success: false, error: 'Invalid segment' });
                            }
                            break;

                        case 'checkSponsorBlockPremium':
                            checkPremiumAccess().then(function(hasPremium) {
                                sendResponse({ hasPremium: hasPremium });
                            });
                            return true; // Keep channel open for async

                        default:
                            debug('Unknown message action:', request.action);
                            break;
                    }
                } catch (e) {
                    error('Message handler error:', e.message);
                    sendResponse({ error: e.message });
                }
                return true; // Keep message channel open for async response
            });

            debug('Message listener set up successfully');
        } catch (e) {
            error('Failed to setup message listener:', e.message);
        }
    }

    // =========================================================================
    // INITIALIZATION
    // =========================================================================

    /**
     * Initialize SponsorBlock integration
     */
    async function init() {
        if (state.initialized) return;

        log('Initializing SponsorBlock integration...');

        // Check premium status first
        const hasPremium = await checkPremiumAccess();

        if (!hasPremium) {
            log('SponsorBlock requires premium - feature disabled');

            // Only run on YouTube video pages for the notice
            if (isVideoPage()) {
                // Show premium notice once after a delay
                setTimeout(function() {
                    showPremiumRequiredNotice();
                }, 3000);
            }

            // Still set up navigation observer to show notice on future video pages
            setupNavigationObserver();

            // Set up message listener for premium status queries
            setupMessageListener();

            return;
        }

        log('Premium verified, activating SponsorBlock');

        // Only run on YouTube video pages
        if (!isVideoPage()) {
            log('Not a video page, waiting for navigation');
            setupNavigationObserver();
            return;
        }

        state.initialized = true;

        // Load user preferences
        await loadCategorySettings();

        // Set up navigation detection for SPA
        setupNavigationObserver();

        // Set up extension communication
        setupMessageListener();

        // Handle current video
        await handleVideoChange();

        log('SponsorBlock initialization complete');
    }

    // =========================================================================
    // STARTUP
    // =========================================================================

    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        // DOM already loaded, initialize immediately
        init();
    }

    // Also try to initialize when YouTube's player is ready
    window.addEventListener('load', function() {
        if (!state.initialized) {
            setTimeout(init, 1000);
        }
    });

    log('SponsorBlock content script loaded (Premium feature)');
})();
