/**
 * StreamShield Premium Features Manager
 * Handles license validation and feature gating
 */

const StreamShieldPremium = {
  // Premium status
  isPremium: false,
  licenseKey: null,
  expiresAt: null,

  // Feature definitions
  FREE_FEATURES: [
    'ad_blocking',
    'popup_blocking',
    'redirect_blocking',
    'youtube_ads',
    'basic_stats',
    'whitelist'
  ],

  PREMIUM_FEATURES: [
    'all_themes',           // 10+ themes (free gets 3)
    'sponsorblock',         // SponsorBlock integration
    'malware_protection',   // Real-time threat blocking
    'phishing_protection',  // Phishing site warnings
    'element_picker',       // Visual element blocker
    'custom_rules',         // Custom filter rules
    'advanced_filters',     // Additional filter lists
    'stats_history',        // Historical statistics
    'stats_export',         // Export stats as CSV/JSON
    'fingerprint_protection', // Anti-fingerprinting
    'priority_support',     // Priority customer support
    // Performance features
    'block_webfonts',       // Block external web fonts
    'block_large_media',    // Block large images/videos
    'lazy_load_images',     // Lazy load off-screen images
    'disable_autoplay',     // Stop video autoplay
    'block_social_widgets', // Block social embeds
    'block_analytics',      // Block analytics scripts
    'block_heavy_scripts',  // Block heavy JS libraries
    'reduced_motion',       // Force reduced motion
    'text_only_mode'        // Extreme text-only mode
  ],

  FREE_THEMES: ['cyber-blue', 'matrix-green', 'pure-light'],
  PREMIUM_THEMES: ['neon-purple', 'sunset-orange', 'midnight-ocean', 'rose-gold', 'arctic-frost', 'forest-emerald', 'dracula'],

  /**
   * Initialize premium system
   */
  async init() {
    await this.loadLicense();
    this.setupStorageListener();
    return this.isPremium;
  },

  /**
   * Load license from storage
   */
  async loadLicense() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['streamshieldLicense'], (result) => {
        if (result.streamshieldLicense) {
          const license = result.streamshieldLicense;
          this.licenseKey = license.key;
          this.expiresAt = license.expiresAt;

          // Check if still valid
          if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
            this.isPremium = true;
          } else if (license.lifetime) {
            this.isPremium = true;
          } else {
            this.isPremium = false;
          }
        }
        resolve(this.isPremium);
      });
    });
  },

  /**
   * Validate a license key
   * Connects to the StreamShield desktop app for license validation
   * This ensures premium/enterprise users get BOTH the extension AND desktop app
   */
  async validateLicense(key) {
    try {
      // Try the local StreamShield desktop app first (port 8081)
      // This links the extension license to the desktop app license
      const response = await fetch('http://127.0.0.1:8081/api/license/validate', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ key: key }) // API expects 'key' not 'license_key'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.valid) {
          // Calculate expires_at from days_remaining
          const expiresAt = data.days_remaining
            ? new Date(Date.now() + data.days_remaining * 24 * 60 * 60 * 1000).toISOString()
            : null;

          const license = {
            key: key,
            validatedAt: new Date().toISOString(),
            expiresAt: expiresAt,
            lifetime: data.tier === 'lifetime' || data.days_remaining > 36500,
            tier: data.tier || 'premium',
            features: data.features || [],
            desktopLinked: true // Flag that this is linked to desktop app
          };

          await this.saveLicense(license);
          this.isPremium = true;
          return { success: true, license, desktopLinked: true };
        }
      }

      return { success: false, error: 'Invalid license key' };
    } catch (e) {
      // Fallback: Accept certain key patterns for offline validation
      if (this.offlineValidate(key)) {
        const license = {
          key: key,
          validatedAt: new Date().toISOString(),
          expiresAt: new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString(), // 1 year
          lifetime: false,
          tier: 'premium',
          offlineValidated: true
        };

        await this.saveLicense(license);
        this.isPremium = true;
        return { success: true, license };
      }

      return { success: false, error: 'Could not validate license' };
    }
  },

  /**
   * Offline validation for specific key patterns
   */
  offlineValidate(key) {
    // Accept keys that match pattern: SS-XXXX-XXXX-XXXX-XXXX
    const pattern = /^SS-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
    return pattern.test(key);
  },

  /**
   * Save license to storage
   */
  async saveLicense(license) {
    return new Promise((resolve) => {
      chrome.storage.sync.set({ streamshieldLicense: license }, resolve);
    });
  },

  /**
   * Remove license
   */
  async removeLicense() {
    this.isPremium = false;
    this.licenseKey = null;
    this.expiresAt = null;
    return new Promise((resolve) => {
      chrome.storage.sync.remove(['streamshieldLicense'], resolve);
    });
  },

  /**
   * Check if a feature is available
   */
  hasFeature(feature) {
    if (this.FREE_FEATURES.includes(feature)) {
      return true;
    }
    if (this.PREMIUM_FEATURES.includes(feature)) {
      return this.isPremium;
    }
    return false;
  },

  /**
   * Check if a theme is available
   */
  hasTheme(themeName) {
    if (this.FREE_THEMES.includes(themeName)) {
      return true;
    }
    if (this.PREMIUM_THEMES.includes(themeName)) {
      return this.isPremium;
    }
    return false;
  },

  /**
   * Get all available themes
   */
  getAvailableThemes() {
    if (this.isPremium) {
      return [...this.FREE_THEMES, ...this.PREMIUM_THEMES];
    }
    return this.FREE_THEMES;
  },

  /**
   * Listen for storage changes
   */
  setupStorageListener() {
    chrome.storage.onChanged.addListener((changes, area) => {
      if (area === 'sync' && changes.streamshieldLicense) {
        this.loadLicense();
      }
    });
  },

  /**
   * Get premium status summary
   */
  getStatus() {
    return {
      isPremium: this.isPremium,
      licenseKey: this.licenseKey ? this.maskKey(this.licenseKey) : null,
      expiresAt: this.expiresAt,
      availableFeatures: this.isPremium
        ? [...this.FREE_FEATURES, ...this.PREMIUM_FEATURES]
        : this.FREE_FEATURES,
      availableThemes: this.getAvailableThemes()
    };
  },

  /**
   * Mask license key for display
   */
  maskKey(key) {
    if (!key || key.length < 8) return '****';
    return key.substring(0, 3) + '-****-****-' + key.substring(key.length - 4);
  },

  /**
   * Show upgrade prompt
   */
  showUpgradePrompt(feature) {
    const featureNames = {
      'all_themes': 'Premium Themes',
      'sponsorblock': 'SponsorBlock',
      'malware_protection': 'Malware Protection',
      'phishing_protection': 'Phishing Protection',
      'element_picker': 'Element Picker',
      'custom_rules': 'Custom Rules',
      'advanced_filters': 'Advanced Filters',
      'stats_history': 'Statistics History',
      'stats_export': 'Export Statistics',
      'fingerprint_protection': 'Fingerprint Protection'
    };

    return {
      title: 'Premium Feature',
      message: `${featureNames[feature] || feature} is a premium feature. Upgrade to unlock all premium features!`,
      upgradeUrl: 'https://streamshield.io/premium',
      features: [
        '10+ Beautiful Themes',
        'SponsorBlock Integration',
        'Malware & Phishing Protection',
        'Element Picker',
        'Custom Filter Rules',
        'Advanced Statistics',
        'Priority Support'
      ]
    };
  }
};

// Export for use in other scripts
if (typeof module !== 'undefined') {
  module.exports = StreamShieldPremium;
}
