/**
 * StreamShield Extension - Popup Controller v4.0
 * Premium Edition with Theme Support & Desktop App Integration
 */

(function() {
  'use strict';

  // Dashboard URL (configurable)
  const DASHBOARD_URL = 'http://127.0.0.1:8081';
  const UPGRADE_URL = 'https://streamshield.io/premium';

  // Default configuration
  let config = {
    enabled: true,
    darkMode: { enabled: false },
    youtube: { skipAds: true, skipSponsors: true },
    adBlocking: { enabled: true },
    privacy: { blockFingerprinting: true, autoRejectCookies: true },
    blockRedirects: true,
    blockPopups: true
  };

  // Stats
  let stats = {
    adsBlocked: 0,
    trackersBlocked: 0,
    dataSaved: 0,
    ytAdsSkipped: 0,
    ytSponsorsSkipped: 0,
    ytTimeSaved: 0,
    threatsBlocked: 0
  };

  // Previous stats for animation
  let prevStats = { ...stats };

  // Premium and desktop app state
  let isPremium = false;
  let isDesktopConnected = false;

  // ========================================================================
  // INITIALIZATION
  // ========================================================================
  document.addEventListener('DOMContentLoaded', async () => {
    loadTheme();
    loadConfig();
    loadStats();
    setupEventListeners();
    checkDarkModeStatus();
    checkWhitelistStatus();

    // Initialize premium system
    await initPremium();

    // Check desktop app connection
    await checkDesktopConnection();

    // Start periodic checks
    setInterval(checkDesktopConnection, 10000); // Check every 10 seconds
  });

  // ========================================================================
  // THEME SUPPORT
  // ========================================================================
  function loadTheme() {
    chrome.storage.sync.get(['streamshieldTheme'], (result) => {
      if (result.streamshieldTheme) {
        document.documentElement.setAttribute('data-theme', result.streamshieldTheme);
      }
    });
  }

  function loadConfig() {
    chrome.storage.sync.get(['streamshieldConfig'], (result) => {
      if (result.streamshieldConfig) {
        config = { ...config, ...result.streamshieldConfig };
        updateUI();
      }
    });
  }

  function saveConfig() {
    chrome.storage.sync.set({ streamshieldConfig: config });
    // Notify content scripts
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, {
          action: 'updateConfig',
          config: config
        }).catch(() => {});
      }
    });
  }

  function loadStats() {
    // Store previous stats for animation
    prevStats = { ...stats };

    chrome.storage.local.get(['streamshieldStats'], (result) => {
      if (result.streamshieldStats) {
        stats = { ...stats, ...result.streamshieldStats };
        updateStatsUI();
      }
    });

    // Also try to get stats from background
    chrome.runtime.sendMessage({ action: 'getStats' }, (response) => {
      if (response) {
        stats.adsBlocked = response.adsBlocked || 0;
        stats.trackersBlocked = response.trackersBlocked || 0;
        stats.threatsBlocked = response.threatsBlocked || 0;
        updateStatsUI();
      }
    });

    // Also try to get stats from desktop app if connected
    if (isDesktopConnected) {
      fetchDesktopStats();
    }
  }

  // ========================================================================
  // PREMIUM SYSTEM
  // ========================================================================
  async function initPremium() {
    try {
      // Check if StreamShieldPremium is available
      if (typeof StreamShieldPremium !== 'undefined') {
        isPremium = await StreamShieldPremium.init();
      } else {
        // Fallback: check storage directly
        const result = await chrome.storage.sync.get(['streamshieldLicense']);
        if (result.streamshieldLicense) {
          const license = result.streamshieldLicense;
          if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
            isPremium = true;
          } else if (license.lifetime) {
            isPremium = true;
          }
        }
      }
    } catch (e) {
      console.debug('Premium init error:', e);
      isPremium = false;
    }

    updatePremiumUI();
  }

  function updatePremiumUI() {
    const premiumBadge = document.getElementById('premium-badge');
    const premiumBadgeText = document.getElementById('premium-badge-text');
    const threatsCard = document.getElementById('threats-card');

    if (premiumBadge && premiumBadgeText) {
      if (isPremium) {
        premiumBadge.classList.remove('free');
        premiumBadge.classList.add('pro');
        premiumBadgeText.textContent = 'PRO';
        premiumBadge.title = 'Premium Active';
      } else {
        premiumBadge.classList.remove('pro');
        premiumBadge.classList.add('free');
        premiumBadgeText.textContent = 'FREE';
        premiumBadge.title = 'Click to upgrade';
      }
    }

    // Update threats card lock status
    if (threatsCard) {
      if (isPremium) {
        threatsCard.classList.remove('premium-locked');
        threatsCard.title = 'Threats Blocked';
      } else {
        threatsCard.classList.add('premium-locked');
        threatsCard.title = 'Premium Feature - Click to Upgrade';
      }
    }
  }

  function showUpgradeModal(feature) {
    const overlay = document.getElementById('upgrade-overlay');
    const message = document.getElementById('upgrade-message');

    if (feature === 'threats') {
      message.textContent = 'Real-time threat protection blocks malware, phishing sites, and dangerous downloads. Upgrade to unlock this powerful security feature!';
    } else {
      message.textContent = 'Unlock all premium features for the ultimate protection experience!';
    }

    if (overlay) {
      overlay.classList.add('show');
    }
  }

  function hideUpgradeModal() {
    const overlay = document.getElementById('upgrade-overlay');
    if (overlay) {
      overlay.classList.remove('show');
    }
  }

  // ========================================================================
  // DESKTOP APP CONNECTION
  // ========================================================================
  async function checkDesktopConnection() {
    // First get connection state from background script (more accurate)
    chrome.runtime.sendMessage({ action: 'getDesktopConnectionStatus' }, async (bgStatus) => {
      if (chrome.runtime.lastError) {
        console.debug('Could not get background status:', chrome.runtime.lastError);
        bgStatus = null;
      }

      // If background reports reconnecting state, show that
      if (bgStatus && (bgStatus.connectionState === 'reconnecting' || bgStatus.connectionState === 'connecting')) {
        isDesktopConnected = false;
        updateDesktopStatusUI(false, bgStatus);
        return;
      }

      try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 3000);

        const response = await fetch(DASHBOARD_URL + '/api/status', {
          method: 'GET',
          signal: controller.signal
        });

        clearTimeout(timeoutId);

        if (response.ok) {
          const data = await response.json();
          isDesktopConnected = true;

          // Merge with background status data
          const statusData = {
            ...data,
            connectionState: 'connected',
            lastSyncTime: bgStatus?.lastSyncTime || new Date().toISOString()
          };

          updateDesktopStatusUI(true, statusData);

          // Get stats from desktop if premium
          if (isPremium && data.stats) {
            stats.threatsBlocked = data.stats.threats_blocked || stats.threatsBlocked;
            updateStatsUI();
          }
        } else {
          isDesktopConnected = false;
          updateDesktopStatusUI(false, bgStatus);
        }
      } catch (e) {
        isDesktopConnected = false;
        updateDesktopStatusUI(false, {
          ...bgStatus,
          lastError: e.message === 'The user aborted a request.' ? 'Connection timeout' : e.message
        });
      }
    });
  }

  async function fetchDesktopStats() {
    try {
      const response = await fetch(DASHBOARD_URL + '/api/stats', {
        method: 'GET'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.threats_blocked !== undefined) {
          stats.threatsBlocked = data.threats_blocked;
        }
        if (data.ads_blocked !== undefined) {
          stats.adsBlocked = Math.max(stats.adsBlocked, data.ads_blocked);
        }
        if (data.trackers_blocked !== undefined) {
          stats.trackersBlocked = Math.max(stats.trackersBlocked, data.trackers_blocked);
        }
        updateStatsUI();
      }
    } catch (e) {
      // Desktop not available
    }
  }

  function updateDesktopStatusUI(connected, data = null) {
    const statusEl = document.getElementById('desktop-status');
    const textEl = document.getElementById('desktop-status-text');
    const detailEl = document.getElementById('desktop-status-detail');
    const reconnectBtn = document.getElementById('desktop-reconnect-btn');

    if (statusEl && textEl) {
      // Reset classes
      statusEl.classList.remove('connected', 'disconnected', 'reconnecting');

      if (data && data.connectionState === 'reconnecting') {
        statusEl.classList.add('reconnecting');
        textEl.textContent = 'Desktop App: Reconnecting...';
        if (detailEl) {
          detailEl.textContent = `Attempt ${data.reconnectAttempts}/${data.maxReconnectAttempts}`;
        }
        if (reconnectBtn) {
          reconnectBtn.disabled = true;
          reconnectBtn.classList.add('spinning');
        }
      } else if (data && data.connectionState === 'connecting') {
        statusEl.classList.add('reconnecting');
        textEl.textContent = 'Desktop App: Connecting...';
        if (detailEl) detailEl.textContent = '';
        if (reconnectBtn) {
          reconnectBtn.disabled = true;
          reconnectBtn.classList.add('spinning');
        }
      } else if (connected) {
        statusEl.classList.add('connected');
        const version = data && data.version ? ` v${data.version}` : '';
        textEl.textContent = `Desktop App: Connected${version}`;
        if (detailEl) {
          detailEl.textContent = data && data.lastSyncTime
            ? `Last sync: ${formatRelativeTime(data.lastSyncTime)}`
            : '';
        }
        if (reconnectBtn) {
          reconnectBtn.disabled = false;
          reconnectBtn.classList.remove('spinning');
          reconnectBtn.title = 'Refresh connection';
        }
      } else {
        statusEl.classList.add('disconnected');
        textEl.textContent = 'Desktop App: Not Connected';
        if (detailEl) {
          detailEl.textContent = data && data.lastError
            ? `Error: ${data.lastError}`
            : 'Click to download or reconnect';
        }
        if (reconnectBtn) {
          reconnectBtn.disabled = false;
          reconnectBtn.classList.remove('spinning');
          reconnectBtn.title = 'Reconnect';
        }
      }
    }
  }

  /**
   * Format relative time
   */
  function formatRelativeTime(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffSecs = Math.floor((now - date) / 1000);

    if (diffSecs < 60) return 'just now';
    if (diffSecs < 3600) return `${Math.floor(diffSecs / 60)}m ago`;
    if (diffSecs < 86400) return `${Math.floor(diffSecs / 3600)}h ago`;
    return date.toLocaleDateString();
  }

  /**
   * Handle reconnect button click
   */
  async function handleReconnect() {
    const reconnectBtn = document.getElementById('desktop-reconnect-btn');
    if (reconnectBtn) {
      reconnectBtn.disabled = true;
      reconnectBtn.classList.add('spinning');
    }

    // Request reconnection from background script
    chrome.runtime.sendMessage({ action: 'forceReconnect' }, (response) => {
      if (chrome.runtime.lastError) {
        console.debug('Reconnect error:', chrome.runtime.lastError);
      }
      // The status will be updated by checkDesktopConnection
      setTimeout(checkDesktopConnection, 500);
    });
  }

  // Current site state
  let currentSite = {
    domain: '',
    isDisabled: false,
    isTrusted: false
  };

  /**
   * Create SVG icon element
   */
  function createSvgIcon(pathData, width = 12, height = 12) {
    const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svg.setAttribute('viewBox', '0 0 24 24');
    svg.setAttribute('width', width.toString());
    svg.setAttribute('height', height.toString());
    svg.setAttribute('fill', 'none');
    svg.setAttribute('stroke', 'currentColor');
    svg.setAttribute('stroke-width', '2');

    pathData.forEach(d => {
      if (d.type === 'path') {
        const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        path.setAttribute('d', d.d);
        svg.appendChild(path);
      } else if (d.type === 'line') {
        const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
        line.setAttribute('x1', d.x1);
        line.setAttribute('y1', d.y1);
        line.setAttribute('x2', d.x2);
        line.setAttribute('y2', d.y2);
        svg.appendChild(line);
      } else if (d.type === 'polyline') {
        const polyline = document.createElementNS('http://www.w3.org/2000/svg', 'polyline');
        polyline.setAttribute('points', d.points);
        svg.appendChild(polyline);
      }
    });

    return svg;
  }

  /**
   * Update current site display in popup
   */
  async function updateCurrentSiteDisplay() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (!tab || !tab.url) {
        updateSiteUI('No site', false, false);
        return;
      }

      // Handle special pages
      if (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://') ||
          tab.url.startsWith('edge://') || tab.url.startsWith('about:')) {
        updateSiteUI('Browser Page', false, false, true);
        return;
      }

      const url = new URL(tab.url);
      currentSite.domain = url.hostname;

      // Check disabled domains
      const localResult = await chrome.storage.local.get(['disabledDomains']);
      const disabledDomains = localResult.disabledDomains || [];
      currentSite.isDisabled = disabledDomains.includes(currentSite.domain);

      // Check whitelist (trusted)
      const [syncResult, localWhitelist] = await Promise.all([
        chrome.storage.sync.get(['whitelist']),
        chrome.storage.local.get(['whitelist'])
      ]);
      const whitelist = [...new Set([...(syncResult.whitelist || []), ...(localWhitelist.whitelist || [])])];
      currentSite.isTrusted = whitelist.includes(currentSite.domain);

      updateSiteUI(currentSite.domain, currentSite.isDisabled, currentSite.isTrusted);
    } catch (e) {
      console.debug('Could not get current site:', e);
      updateSiteUI('Unknown', false, false);
    }
  }

  /**
   * Update the site UI elements
   */
  function updateSiteUI(domain, isDisabled, isTrusted, isSpecialPage = false) {
    const domainEl = document.getElementById('current-domain');
    const statusEl = document.getElementById('site-status');
    const disableBtn = document.getElementById('disable-site-btn');
    const trustBtn = document.getElementById('trust-site-btn');

    if (domainEl) {
      domainEl.textContent = domain.length > 25 ? domain.substring(0, 22) + '...' : domain;
      domainEl.title = domain;
    }

    if (isSpecialPage) {
      if (statusEl) {
        statusEl.textContent = 'N/A';
        statusEl.style.color = 'var(--text-muted)';
      }
      if (disableBtn) disableBtn.style.display = 'none';
      if (trustBtn) trustBtn.style.display = 'none';
      return;
    }

    if (statusEl) {
      if (isDisabled) {
        statusEl.textContent = 'Disabled';
        statusEl.style.color = 'var(--cyber-red)';
      } else if (isTrusted) {
        statusEl.textContent = 'Trusted';
        statusEl.style.color = 'var(--cyber-orange)';
      } else {
        statusEl.textContent = 'Protected';
        statusEl.style.color = 'var(--cyber-green)';
      }
    }

    // Update button states using safe DOM methods
    if (disableBtn) {
      disableBtn.style.display = 'flex';
      // Clear existing content
      while (disableBtn.firstChild) {
        disableBtn.removeChild(disableBtn.firstChild);
      }

      if (isDisabled) {
        // Enable icon (checkmark)
        const icon = createSvgIcon([{ type: 'polyline', points: '20 6 9 17 4 12' }]);
        disableBtn.appendChild(icon);
        disableBtn.appendChild(document.createTextNode(' Enable'));
        disableBtn.title = 'Enable protection on this site';
      } else {
        // Disable icon (power)
        const icon = createSvgIcon([
          { type: 'path', d: 'M18.36 6.64a9 9 0 1 1-12.73 0' },
          { type: 'line', x1: '12', y1: '2', x2: '12', y2: '12' }
        ]);
        disableBtn.appendChild(icon);
        disableBtn.appendChild(document.createTextNode(' Disable'));
        disableBtn.title = 'Disable protection on this site';
      }
    }

    if (trustBtn) {
      trustBtn.style.display = 'flex';
      // Clear existing content
      while (trustBtn.firstChild) {
        trustBtn.removeChild(trustBtn.firstChild);
      }

      if (isTrusted) {
        // Untrust icon (X)
        const icon = createSvgIcon([
          { type: 'line', x1: '18', y1: '6', x2: '6', y2: '18' },
          { type: 'line', x1: '6', y1: '6', x2: '18', y2: '18' }
        ]);
        trustBtn.appendChild(icon);
        trustBtn.appendChild(document.createTextNode(' Untrust'));
        trustBtn.title = 'Remove from trusted sites';
      } else {
        // Trust icon (check circle)
        const icon = createSvgIcon([
          { type: 'path', d: 'M22 11.08V12a10 10 0 1 1-5.93-9.14' },
          { type: 'polyline', points: '22 4 12 14.01 9 11.01' }
        ]);
        trustBtn.appendChild(icon);
        trustBtn.appendChild(document.createTextNode(' Trust'));
        trustBtn.title = 'Always trust this site';
      }
    }
  }

  /**
   * Handle disable/enable site
   */
  async function handleDisableSite() {
    if (!currentSite.domain) return;

    try {
      const result = await chrome.storage.local.get(['disabledDomains']);
      let disabledDomains = result.disabledDomains || [];

      if (currentSite.isDisabled) {
        // Enable (remove from disabled list)
        disabledDomains = disabledDomains.filter(d => d !== currentSite.domain);
        currentSite.isDisabled = false;
      } else {
        // Disable
        if (!disabledDomains.includes(currentSite.domain)) {
          disabledDomains.push(currentSite.domain);
        }
        currentSite.isDisabled = true;
      }

      await chrome.storage.local.set({ disabledDomains });

      // Notify background script
      chrome.runtime.sendMessage({
        action: currentSite.isDisabled ? 'disableSite' : 'enableSite',
        domain: currentSite.domain
      });

      // Notify content script
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab) {
        chrome.tabs.sendMessage(tab.id, {
          action: 'toggleEnabled',
          enabled: !currentSite.isDisabled
        }).catch(() => {});
      }

      updateSiteUI(currentSite.domain, currentSite.isDisabled, currentSite.isTrusted);
    } catch (e) {
      console.error('Error toggling site:', e);
    }
  }

  /**
   * Handle trust/untrust site
   */
  async function handleTrustSite() {
    if (!currentSite.domain) return;

    try {
      // Get current whitelist from both storages
      const [syncResult, localResult] = await Promise.all([
        chrome.storage.sync.get(['whitelist']),
        chrome.storage.local.get(['whitelist'])
      ]);

      let whitelist = [...new Set([...(syncResult.whitelist || []), ...(localResult.whitelist || [])])];

      if (currentSite.isTrusted) {
        // Untrust (remove from whitelist)
        whitelist = whitelist.filter(d => d !== currentSite.domain);
        currentSite.isTrusted = false;

        // Notify background
        chrome.runtime.sendMessage({ action: 'removeFromWhitelist', domain: currentSite.domain });
      } else {
        // Trust (add to whitelist)
        if (!whitelist.includes(currentSite.domain)) {
          whitelist.push(currentSite.domain);
        }
        currentSite.isTrusted = true;

        // Notify background
        chrome.runtime.sendMessage({ action: 'addToWhitelist', domain: currentSite.domain });
      }

      // Save to both storages
      await Promise.all([
        chrome.storage.sync.set({ whitelist }),
        chrome.storage.local.set({ whitelist })
      ]);

      // Try to sync to proxy
      syncWhitelistToProxy(whitelist);

      updateSiteUI(currentSite.domain, currentSite.isDisabled, currentSite.isTrusted);
    } catch (e) {
      console.error('Error toggling trust:', e);
    }
  }

  // ========================================================================
  // THEME TOGGLE - Light/Dark mode quick toggle
  // ========================================================================

  let isLightMode = false;
  let previousTheme = null;

  /**
   * Initialize theme toggle state from storage
   */
  async function initializeThemeToggle() {
    try {
      const result = await chrome.storage.sync.get(['themePreference', 'selectedTheme']);
      const currentTheme = result.selectedTheme || 'cyber-blue';

      isLightMode = currentTheme === 'pure-light';

      // Apply theme to popup
      if (isLightMode) {
        document.documentElement.setAttribute('data-theme', 'pure-light');
      }

      // Update toggle button state
      updateThemeToggleIcon();
    } catch (e) {
      console.debug('Could not load theme preference:', e);
    }
  }

  /**
   * Update the theme toggle icon based on current mode
   */
  function updateThemeToggleIcon() {
    const darkIcon = document.querySelector('.theme-icon-dark');
    const lightIcon = document.querySelector('.theme-icon-light');

    if (darkIcon && lightIcon) {
      if (isLightMode) {
        darkIcon.style.display = 'none';
        lightIcon.style.display = 'block';
      } else {
        darkIcon.style.display = 'block';
        lightIcon.style.display = 'none';
      }
    }
  }

  /**
   * Toggle between light and dark mode
   */
  async function toggleLightDarkMode() {
    try {
      const result = await chrome.storage.sync.get(['selectedTheme']);
      const currentTheme = result.selectedTheme || 'cyber-blue';

      if (isLightMode) {
        // Switch to dark mode (use previous theme or default)
        const newTheme = previousTheme || 'cyber-blue';
        document.documentElement.setAttribute('data-theme', newTheme);
        await chrome.storage.sync.set({ selectedTheme: newTheme });
        isLightMode = false;
      } else {
        // Switch to light mode
        previousTheme = currentTheme;
        document.documentElement.setAttribute('data-theme', 'pure-light');
        await chrome.storage.sync.set({
          selectedTheme: 'pure-light',
          previousDarkTheme: currentTheme
        });
        isLightMode = true;
      }

      updateThemeToggleIcon();

      // Notify content scripts to update theme
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]) {
          chrome.tabs.sendMessage(tabs[0].id, {
            action: 'updateTheme',
            theme: isLightMode ? 'pure-light' : (previousTheme || 'cyber-blue')
          }).catch(() => {});
        }
      });

    } catch (e) {
      console.error('Error toggling theme:', e);
    }
  }

  // ========================================================================
  // UI UPDATES
  // ========================================================================
  function updateUI() {
    // Protection toggle
    const protectionCard = document.getElementById('toggle-protection');
    if (protectionCard) {
      protectionCard.classList.toggle('active', config.enabled);
      protectionCard.querySelector('.toggle-status').textContent = config.enabled ? 'Active' : 'Off';
    }

    // Redirects toggle
    const redirectsCard = document.getElementById('toggle-redirects');
    if (redirectsCard) {
      redirectsCard.classList.toggle('active', config.blockRedirects);
      redirectsCard.querySelector('.toggle-status').textContent = config.blockRedirects ? 'Active' : 'Off';
    }

    // Popups toggle
    const popupsCard = document.getElementById('toggle-popups');
    if (popupsCard) {
      popupsCard.classList.toggle('active', config.blockPopups);
      popupsCard.querySelector('.toggle-status').textContent = config.blockPopups ? 'Active' : 'Off';
    }

    // Dark mode toggle
    const darkModeCard = document.getElementById('toggle-darkmode');
    if (darkModeCard) {
      darkModeCard.classList.toggle('active', config.darkMode.enabled);
      darkModeCard.querySelector('.toggle-status').textContent = config.darkMode.enabled ? 'On' : 'Off';
    }

    // YouTube toggle
    const youtubeCard = document.getElementById('toggle-youtube');
    if (youtubeCard) {
      youtubeCard.classList.toggle('active', config.youtube.skipAds);
      youtubeCard.querySelector('.toggle-status').textContent = config.youtube.skipAds ? 'Ad Skip On' : 'Off';
    }

    // Privacy toggle
    const privacyCard = document.getElementById('toggle-privacy');
    if (privacyCard) {
      privacyCard.classList.toggle('active', config.privacy.blockFingerprinting);
      privacyCard.querySelector('.toggle-status').textContent = config.privacy.blockFingerprinting ? 'Protected' : 'Off';
    }
  }

  function updateStatsUI() {
    const adsBlocked = document.getElementById('ads-blocked');
    const trackersBlocked = document.getElementById('trackers-blocked');
    const dataSaved = document.getElementById('data-saved');
    const threatsBlocked = document.getElementById('threats-blocked');
    const ytAdsSkipped = document.getElementById('yt-ads-skipped');
    const ytSponsorsSkipped = document.getElementById('yt-sponsors-skipped');
    const ytTimeSaved = document.getElementById('yt-time-saved');

    // Animate if values changed
    animateCounter(adsBlocked, prevStats.adsBlocked, stats.adsBlocked);
    animateCounter(trackersBlocked, prevStats.trackersBlocked, stats.trackersBlocked);
    animateCounter(threatsBlocked, prevStats.threatsBlocked, stats.threatsBlocked);

    if (dataSaved) dataSaved.textContent = formatBytes(stats.dataSaved);
    if (ytAdsSkipped) ytAdsSkipped.textContent = formatNumber(stats.ytAdsSkipped);
    if (ytSponsorsSkipped) ytSponsorsSkipped.textContent = formatNumber(stats.ytSponsorsSkipped);
    if (ytTimeSaved) ytTimeSaved.textContent = formatTime(stats.ytTimeSaved);

    // Update previous stats
    prevStats = { ...stats };
  }

  function animateCounter(element, fromValue, toValue) {
    if (!element) return;

    // If no change, just set the value
    if (fromValue === toValue) {
      element.textContent = formatNumber(toValue);
      return;
    }

    // Animate the counter
    const duration = 500; // ms
    const startTime = performance.now();
    const diff = toValue - fromValue;

    // Add animation class
    element.classList.add('counting');

    function update(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);

      // Easing function (ease-out)
      const easeProgress = 1 - Math.pow(1 - progress, 3);

      const currentValue = Math.round(fromValue + (diff * easeProgress));
      element.textContent = formatNumber(currentValue);

      if (progress < 1) {
        requestAnimationFrame(update);
      } else {
        element.textContent = formatNumber(toValue);
        // Remove animation class after a short delay
        setTimeout(() => element.classList.remove('counting'), 100);
      }
    }

    requestAnimationFrame(update);
  }

  function formatNumber(n) {
    if (!n) return '0';
    if (n >= 1000000) return (n / 1000000).toFixed(1) + 'M';
    if (n >= 1000) return (n / 1000).toFixed(1) + 'K';
    return n.toString();
  }

  function formatBytes(bytes) {
    if (!bytes) return '0';
    if (bytes >= 1073741824) return (bytes / 1073741824).toFixed(1) + 'GB';
    if (bytes >= 1048576) return (bytes / 1048576).toFixed(1);
    if (bytes >= 1024) return (bytes / 1024).toFixed(1) + 'KB';
    return bytes + 'B';
  }

  function formatTime(seconds) {
    if (!seconds) return '0m';
    if (seconds >= 3600) return Math.floor(seconds / 3600) + 'h';
    if (seconds >= 60) return Math.floor(seconds / 60) + 'm';
    return seconds + 's';
  }

  // ========================================================================
  // EVENT LISTENERS
  // ========================================================================
  function setupEventListeners() {
    // Protection toggle
    const protectionToggle = document.getElementById('toggle-protection');
    if (protectionToggle) {
      protectionToggle.addEventListener('click', () => {
        config.enabled = !config.enabled;
        config.adBlocking.enabled = config.enabled;
        saveConfig();
        updateUI();
      });
    }

    // Dark mode toggle
    const darkModeToggle = document.getElementById('toggle-darkmode');
    if (darkModeToggle) {
      darkModeToggle.addEventListener('click', () => {
        config.darkMode.enabled = !config.darkMode.enabled;
        saveConfig();
        updateUI();

        // Toggle dark mode on current tab
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs[0]) {
            chrome.tabs.sendMessage(tabs[0].id, { action: 'toggleDarkMode' }).catch(() => {});
          }
        });
      });
    }

    // YouTube toggle
    const youtubeToggle = document.getElementById('toggle-youtube');
    if (youtubeToggle) {
      youtubeToggle.addEventListener('click', () => {
        config.youtube.skipAds = !config.youtube.skipAds;
        config.youtube.skipSponsors = config.youtube.skipAds;
        saveConfig();
        updateUI();
      });
    }

    // Privacy toggle
    const privacyToggle = document.getElementById('toggle-privacy');
    if (privacyToggle) {
      privacyToggle.addEventListener('click', () => {
        config.privacy.blockFingerprinting = !config.privacy.blockFingerprinting;
        config.privacy.autoRejectCookies = config.privacy.blockFingerprinting;
        saveConfig();
        updateUI();
      });
    }

    // Redirects toggle
    const redirectsToggle = document.getElementById('toggle-redirects');
    if (redirectsToggle) {
      redirectsToggle.addEventListener('click', () => {
        config.blockRedirects = !config.blockRedirects;
        saveConfig();
        updateUI();
        // Notify background script
        chrome.runtime.sendMessage({ action: 'updateConfig', config: config });
      });
    }

    // Popups toggle
    const popupsToggle = document.getElementById('toggle-popups');
    if (popupsToggle) {
      popupsToggle.addEventListener('click', () => {
        config.blockPopups = !config.blockPopups;
        saveConfig();
        updateUI();
        // Notify background script
        chrome.runtime.sendMessage({ action: 'updateConfig', config: config });
      });
    }

    // Dashboard button
    const dashboardBtn = document.getElementById('dashboard-btn');
    if (dashboardBtn) {
      dashboardBtn.addEventListener('click', () => {
        chrome.tabs.create({ url: DASHBOARD_URL });
      });
    }

    // Whitelist button
    const whitelistBtn = document.getElementById('whitelist-btn');
    if (whitelistBtn) {
      whitelistBtn.addEventListener('click', toggleWhitelist);
    }

    // Report Broken Site button
    const reportBrokenBtn = document.getElementById('report-broken-btn');
    if (reportBrokenBtn) {
      reportBrokenBtn.addEventListener('click', async () => {
        try {
          // Get current tab URL
          const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
          if (!tabs[0]) return;

          const url = tabs[0].url;
          const domain = new URL(url).hostname;

          // Show loading state
          reportBrokenBtn.disabled = true;
          reportBrokenBtn.textContent = 'Reporting...';

          // Send report to compat API
          const response = await fetch(`${DASHBOARD_URL}/api/compat/report`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
              domain: domain,
              url: url,
              category: 'unknown',
              description: 'User reported broken page from extension',
              user_agent: navigator.userAgent
            })
          });

          const result = await response.json();

          if (result.success) {
            // Show success message
            document.getElementById('report-success').style.display = 'block';
            reportBrokenBtn.style.display = 'none';

            // Hide success message after 5 seconds
            setTimeout(() => {
              document.getElementById('report-success').style.display = 'none';
              reportBrokenBtn.style.display = 'flex';
              reportBrokenBtn.disabled = false;
              reportBrokenBtn.textContent = 'Report Broken Site';
            }, 5000);
          } else {
            throw new Error(result.error || 'Unknown error');
          }
        } catch (error) {
          console.error('Failed to report broken site:', error);
          reportBrokenBtn.disabled = false;
          reportBrokenBtn.textContent = 'Report Failed - Retry';
          setTimeout(() => {
            reportBrokenBtn.textContent = 'Report Broken Site';
          }, 3000);
        }
      });
    }

    // Settings button
    const settingsBtn = document.getElementById('settings-btn');
    if (settingsBtn) {
      settingsBtn.addEventListener('click', () => {
        if (chrome.runtime.openOptionsPage) {
          chrome.runtime.openOptionsPage();
        }
      });
    }

    // Help link
    const helpLink = document.getElementById('help-link');
    if (helpLink) {
      helpLink.addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ url: 'https://github.com/streamshield/help' });
      });
    }

    // 5-minute pause button
    const pause5mBtn = document.getElementById('pause-5m');
    if (pause5mBtn) {
      pause5mBtn.addEventListener('click', () => pauseBlocking(5));
    }

    // Premium badge click
    const premiumBadge = document.getElementById('premium-badge');
    if (premiumBadge) {
      premiumBadge.addEventListener('click', () => {
        if (!isPremium) {
          showUpgradeModal('general');
        } else {
          // Show premium status info or open settings
          chrome.tabs.create({ url: DASHBOARD_URL + '/license' });
        }
      });
    }

    // Threats card click (upgrade prompt if not premium)
    const threatsCard = document.getElementById('threats-card');
    if (threatsCard) {
      threatsCard.addEventListener('click', () => {
        if (!isPremium) {
          showUpgradeModal('threats');
        }
      });
    }

    // Upgrade modal close button
    const upgradeClose = document.getElementById('upgrade-close');
    if (upgradeClose) {
      upgradeClose.addEventListener('click', hideUpgradeModal);
    }

    // Upgrade modal overlay click to close
    const upgradeOverlay = document.getElementById('upgrade-overlay');
    if (upgradeOverlay) {
      upgradeOverlay.addEventListener('click', (e) => {
        if (e.target === upgradeOverlay) {
          hideUpgradeModal();
        }
      });
    }

    // Upgrade button
    const upgradeBtn = document.getElementById('upgrade-btn');
    if (upgradeBtn) {
      upgradeBtn.addEventListener('click', () => {
        chrome.tabs.create({ url: UPGRADE_URL });
        hideUpgradeModal();
      });
    }

    // Desktop status click - open dashboard or download page
    const desktopStatus = document.getElementById('desktop-status');
    if (desktopStatus) {
      desktopStatus.addEventListener('click', (e) => {
        // Don't trigger if clicking reconnect button
        if (e.target.closest('#desktop-reconnect-btn')) return;

        if (isDesktopConnected) {
          chrome.tabs.create({ url: DASHBOARD_URL });
        } else {
          chrome.tabs.create({ url: 'https://streamshield.io/download' });
        }
      });
    }

    // Reconnect button click
    const reconnectBtn = document.getElementById('desktop-reconnect-btn');
    if (reconnectBtn) {
      reconnectBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        handleReconnect();
      });
    }

    // Disable site button
    const disableSiteBtn = document.getElementById('disable-site-btn');
    if (disableSiteBtn) {
      disableSiteBtn.addEventListener('click', handleDisableSite);
    }

    // Trust site button
    const trustSiteBtn = document.getElementById('trust-site-btn');
    if (trustSiteBtn) {
      trustSiteBtn.addEventListener('click', handleTrustSite);
    }

    // Initialize current site display
    updateCurrentSiteDisplay();

    // Theme toggle button
    const themeToggleBtn = document.getElementById('theme-toggle-btn');
    if (themeToggleBtn) {
      themeToggleBtn.addEventListener('click', toggleLightDarkMode);
    }

    // Initialize theme toggle state
    initializeThemeToggle();

    // Open Settings link in footer
    const openSettingsLink = document.getElementById('open-settings-link');
    if (openSettingsLink) {
      openSettingsLink.addEventListener('click', (e) => {
        e.preventDefault();
        if (chrome.runtime.openOptionsPage) {
          chrome.runtime.openOptionsPage();
        } else {
          chrome.tabs.create({ url: chrome.runtime.getURL('options.html') });
        }
      });
    }

    // Start live stats updates
    startLiveStatsUpdates();
  }

  // ========================================================================
  // REPORT BROKEN SITE
  // ========================================================================
  async function reportBrokenSite() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (!tab || !tab.url) return;

      const url = new URL(tab.url);
      const domain = url.hostname;
      const fullUrl = tab.url;

      // Collect report data
      const reportData = {
        domain: domain,
        url: fullUrl,
        timestamp: new Date().toISOString(),
        userAgent: navigator.userAgent,
        extensionVersion: '4.0',
        config: config,
        stats: stats
      };

      // Try to send to backend
      try {
        const response = await fetch(DASHBOARD_URL + '/api/compat/report', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(reportData)
        });
        if (response.ok) {
          console.log('Report sent to backend');
        }
      } catch (e) {
        console.debug('Backend not available, storing locally');
      }

      // Store report locally
      chrome.storage.local.get(['brokenSiteReports'], (result) => {
        const reports = result.brokenSiteReports || [];
        reports.push(reportData);
        // Keep last 50 reports
        if (reports.length > 50) reports.shift();
        chrome.storage.local.set({ brokenSiteReports: reports });
      });

      // Auto-whitelist the site temporarily
      chrome.storage.local.get(['whitelist'], (result) => {
        const whitelist = result.whitelist || [];
        if (!whitelist.includes(domain)) {
          whitelist.push(domain);
          chrome.storage.local.set({ whitelist });
          // Notify background
          chrome.runtime.sendMessage({ action: 'addToWhitelist', domain: domain });
        }
      });

      // Show success message
      const reportBtn = document.getElementById('report-broken-btn');
      const successMsg = document.getElementById('report-success');
      if (reportBtn) reportBtn.style.display = 'none';
      if (successMsg) successMsg.style.display = 'block';

      // Reset after 3 seconds
      setTimeout(() => {
        if (reportBtn) reportBtn.style.display = 'flex';
        if (successMsg) successMsg.style.display = 'none';
      }, 3000);

    } catch (e) {
      console.error('Report error:', e);
    }
  }

  // ========================================================================
  // PAUSE BLOCKING
  // ========================================================================
  function pauseBlocking(minutes) {
    const endTime = Date.now() + (minutes * 60 * 1000);
    chrome.storage.local.set({ pauseEndTime: endTime });
    chrome.runtime.sendMessage({ action: 'pauseBlocking', duration: minutes });
    updatePauseUI(endTime);
  }

  function updatePauseUI(endTime) {
    const pauseStatus = document.getElementById('pause-status');
    const resumeBtn = document.getElementById('resume-btn');
    const countdown = document.getElementById('pause-countdown');

    if (pauseStatus) pauseStatus.style.display = 'block';
    if (resumeBtn) resumeBtn.style.display = 'block';

    const updateCountdown = () => {
      const remaining = endTime - Date.now();
      if (remaining <= 0) {
        if (pauseStatus) pauseStatus.style.display = 'none';
        if (resumeBtn) resumeBtn.style.display = 'none';
        return;
      }
      const mins = Math.floor(remaining / 60000);
      const secs = Math.floor((remaining % 60000) / 1000);
      if (countdown) countdown.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
      setTimeout(updateCountdown, 1000);
    };
    updateCountdown();
  }

  // ========================================================================
  // LIVE STATS UPDATES
  // ========================================================================
  function startLiveStatsUpdates() {
    // Update stats every 2 seconds
    setInterval(() => {
      loadStats();
      updateLastUpdateTime();
    }, 2000);
  }

  function updateLastUpdateTime() {
    const lastUpdate = document.getElementById('last-update');
    if (lastUpdate) {
      lastUpdate.textContent = 'Updated just now';
    }
  }

  // ========================================================================
  // WHITELIST
  // ========================================================================
  async function toggleWhitelist() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && tab.url) {
        const url = new URL(tab.url);
        const domain = url.hostname;

        // Get current whitelist from both storages
        const [syncResult, localResult] = await Promise.all([
          chrome.storage.sync.get(['whitelist']),
          chrome.storage.local.get(['whitelist'])
        ]);

        // Merge whitelists
        const syncWhitelist = syncResult.whitelist || [];
        const localWhitelist = localResult.whitelist || [];
        let whitelist = [...new Set([...syncWhitelist, ...localWhitelist])];
        const whitelistBtn = document.getElementById('whitelist-btn');

        if (whitelist.includes(domain)) {
          // Remove from whitelist
          whitelist = whitelist.filter(d => d !== domain);
          if (whitelistBtn) whitelistBtn.textContent = '✅ Whitelist This Site';
          // Notify background to remove
          chrome.runtime.sendMessage({ action: 'removeFromWhitelist', domain: domain });
        } else {
          // Add to whitelist
          whitelist.push(domain);
          if (whitelistBtn) whitelistBtn.textContent = '❌ Remove from Whitelist';
          // Notify background to add
          chrome.runtime.sendMessage({ action: 'addToWhitelist', domain: domain });
        }

        // Save to both storages
        await Promise.all([
          chrome.storage.sync.set({ whitelist }),
          chrome.storage.local.set({ whitelist })
        ]);

        // Try to sync to proxy
        syncWhitelistToProxy(whitelist);
      }
    } catch (e) {
      console.error('Whitelist error:', e);
    }
  }

  // Sync whitelist to proxy if available
  async function syncWhitelistToProxy(whitelist) {
    try {
      const response = await fetch(DASHBOARD_URL + '/api/extension/config', {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ whitelist: whitelist })
      });
      if (response.ok) {
        console.log('Whitelist synced to proxy');
      }
    } catch (e) {
      // Proxy not running - OK, extension works standalone
      console.debug('Proxy sync skipped');
    }
  }

  async function checkWhitelistStatus() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && tab.url) {
        const url = new URL(tab.url);
        const domain = url.hostname;

        // Check both storages
        const [syncResult, localResult] = await Promise.all([
          chrome.storage.sync.get(['whitelist']),
          chrome.storage.local.get(['whitelist'])
        ]);

        const syncWhitelist = syncResult.whitelist || [];
        const localWhitelist = localResult.whitelist || [];
        const whitelist = [...new Set([...syncWhitelist, ...localWhitelist])];
        const whitelistBtn = document.getElementById('whitelist-btn');

        if (whitelist.includes(domain) && whitelistBtn) {
          whitelistBtn.textContent = '❌ Remove from Whitelist';
        }
      }
    } catch (e) {
      // Ignore errors for special pages
    }
  }

  // ========================================================================
  // DARK MODE STATUS
  // ========================================================================
  function checkDarkModeStatus() {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, { action: 'getDarkModeStatus' }, (response) => {
          if (chrome.runtime.lastError) {
            // Tab might not have content script
            return;
          }
          if (response && response.darkModeEnabled !== undefined) {
            config.darkMode.enabled = response.darkModeEnabled;
            updateUI();
          }
        });
      }
    });
  }

})();
