/**
 * StreamShield Pause Blocker Controller
 * Manages pause/resume blocking UI and API communication
 */

// Dashboard/API URL
const DASHBOARD_URL = 'http://127.0.0.1:8081';

// API token for authenticated requests (empty for localhost)
const API_TOKEN = '';

class PauseBlocker {
    constructor() {
        this.pauseService = null;
        this.countdownInterval = null;
        this.init();
    }

    async init() {
        console.log('⏸️  Initializing Pause Blocker...');

        // Initialize pause status on page load
        await this.updatePauseStatus();

        // Set up event listeners for extension popup (if present)
        this.setupExtensionListeners();

        // Set up event listeners for dashboard (if present)
        this.setupDashboardListeners();

        // Poll pause status every 5 seconds
        setInterval(() => this.updatePauseStatus(), 5000);

        console.log('✅ Pause Blocker initialized');
    }

    setupExtensionListeners() {
        // Pause buttons in extension popup
        const pauseButtons = {
            '1h': document.getElementById('pause-1h'),
            '4h': document.getElementById('pause-4h'),
            '8h': document.getElementById('pause-8h'),
            '24h': document.getElementById('pause-24h')
        };

        const hoursMap = {
            '1h': 1,
            '4h': 4,
            '8h': 8,
            '24h': 24
        };

        for (const [key, btn] of Object.entries(pauseButtons)) {
            if (btn) {
                btn.addEventListener('click', () => {
                    this.pauseBlocking(hoursMap[key], `Paused for ${key}`);
                });
            }
        }

        // Resume button in extension popup
        const resumeBtn = document.getElementById('resume-btn');
        if (resumeBtn) {
            resumeBtn.addEventListener('click', () => this.resumeBlocking());
        }
    }

    setupDashboardListeners() {
        // Resume button in dashboard alert
        const resumeNowBtn = document.getElementById('resume-now-btn');
        if (resumeNowBtn) {
            resumeNowBtn.addEventListener('click', () => this.resumeBlocking());
        }
    }

    async pauseBlocking(hours, reason = null) {
        console.log(`⏸️  Pausing blocking for ${hours} hour(s)...`);

        try {
            const response = await fetch(`${DASHBOARD_URL}/api/blocking/pause`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${API_TOKEN || ''}`
                },
                body: JSON.stringify({
                    hours: hours,
                    reason: reason || `User paused blocking for ${hours} hour(s)`
                })
            });

            const data = await response.json();

            if (data.success) {
                console.log('✅ Blocking paused successfully');
                this.showNotification(`Blocking paused for ${hours} hour(s)`, 'success');
                await this.updatePauseStatus();
            } else {
                console.error('❌ Failed to pause blocking:', data.message);
                this.showNotification(`Error: ${data.message}`, 'error');
            }
        } catch (error) {
            console.error('Error pausing blocking:', error);
            this.showNotification('Error pausing blocking', 'error');
        }
    }

    async resumeBlocking() {
        console.log('▶️  Resuming blocking...');

        try {
            const response = await fetch(`${DASHBOARD_URL}/api/blocking/resume`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${API_TOKEN || ''}`
                },
                body: JSON.stringify({
                    reason: 'User manually resumed blocking'
                })
            });

            const data = await response.json();

            if (data.success) {
                console.log('✅ Blocking resumed successfully');
                this.showNotification('Blocking resumed', 'success');
                await this.updatePauseStatus();
            } else {
                console.error('❌ Failed to resume blocking:', data.message);
                this.showNotification(`Error: ${data.message}`, 'error');
            }
        } catch (error) {
            console.error('Error resuming blocking:', error);
            this.showNotification('Error resuming blocking', 'error');
        }
    }

    async updatePauseStatus() {
        try {
            const response = await fetch(`${DASHBOARD_URL}/api/blocking/pause-status`, {
                headers: {
                    'Authorization': `Bearer ${API_TOKEN || ''}`
                }
            });

            const status = await response.json();

            // Update extension popup UI
            this.updateExtensionUI(status);

            // Update dashboard UI
            this.updateDashboardUI(status);

        } catch (error) {
            console.error('Error updating pause status:', error);
        }
    }

    updateExtensionUI(status) {
        const pauseStatus = document.getElementById('pause-status');
        const resumeBtn = document.getElementById('resume-btn');
        const pauseCountdown = document.getElementById('pause-countdown');

        if (pauseStatus && status.is_paused) {
            pauseStatus.style.display = 'block';
            if (resumeBtn) resumeBtn.style.display = 'block';

            // Update countdown
            if (pauseCountdown) {
                pauseCountdown.textContent = this.formatMinutes(status.remaining_minutes);
            }
        } else if (pauseStatus) {
            pauseStatus.style.display = 'none';
            if (resumeBtn) resumeBtn.style.display = 'none';
        }
    }

    updateDashboardUI(status) {
        const pauseAlert = document.getElementById('pause-status-alert');
        const pauseCountdown = document.getElementById('pause-countdown-dashboard');

        if (pauseAlert && status.is_paused) {
            pauseAlert.style.display = 'block';

            // Update countdown
            if (pauseCountdown) {
                pauseCountdown.textContent = this.formatMinutes(status.remaining_minutes);
            }
        } else if (pauseAlert) {
            pauseAlert.style.display = 'none';
        }
    }

    formatMinutes(minutes) {
        const hours = Math.floor(minutes / 60);
        const mins = minutes % 60;

        if (hours > 0) {
            return `${hours}h ${mins}m`;
        } else {
            return `${mins}m`;
        }
    }

    showNotification(message, type = 'info') {
        // Simple notification - could be enhanced with toast library
        const color = type === 'success' ? '#00FF88' : type === 'error' ? '#FF3366' : '#00F0FF';
        console.log(`%c${message}`, `color: ${color}; font-weight: bold;`);

        // Optional: Create visual toast notification
        if (typeof showToast === 'function') {
            showToast(message, type);
        }
    }
}

// Initialize pause blocker on page load
document.addEventListener('DOMContentLoaded', () => {
    window.pauseBlocker = new PauseBlocker();
});

// Also initialize if page already loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.pauseBlocker = new PauseBlocker();
    });
} else {
    window.pauseBlocker = new PauseBlocker();
}
