/**
 * StreamShield Options Page JavaScript
 * Handles all settings, storage, and UI interactions for the options page
 *
 * @version 4.0.0
 */

// ============================================
// STORAGE KEYS
// ============================================
const STORAGE_KEYS = {
  CONFIG: 'streamshieldConfig',
  THEME: 'streamshieldTheme',
  STATS: 'streamshieldStats',
  RULES: 'streamshieldRules',
  WHITELIST: 'whitelist',
  FILTER_LISTS: 'filterLists',
  SPONSORBLOCK: 'sponsorblockCategories',
  LICENSE: 'streamshieldLicense'
};

// Desktop App Configuration
const DESKTOP_APP_CONFIG = {
  baseUrl: 'http://localhost:8081',
  endpoints: {
    health: '/api/health',
    license: '/api/license',
    licenseSync: '/api/license/sync',
    licenseActivate: '/api/license/activate'
  },
  timeout: 3000
};

// ============================================
// DEFAULT SETTINGS
// ============================================
const DEFAULT_CONFIG = {
  // General
  protection: true,
  blockAds: true,
  blockPopups: true,
  blockRedirects: true,
  showBadge: true,
  threatAlerts: true,

  // Appearance
  animations: true,
  compactMode: false,

  // YouTube
  youtubeAds: true,
  youtubeShorts: false,
  youtubeDistractionFree: false,
  sponsorblock: true,

  // Security
  malwareProtection: true,
  phishingProtection: true,
  cryptominerBlocking: true,
  scamProtection: true,

  // Privacy
  blockTrackers: true,
  fingerprintProtection: true,
  webrtcLeak: false,
  autoRejectCookies: true,
  hideCookieNotices: true,

  // Performance & Speed Optimizations (Premium)
  blockWebFonts: false,         // Block external web fonts (faster page loads)
  blockLargeMedia: false,       // Block images/videos over size threshold
  lazyLoadImages: true,         // Lazy load off-screen images
  disableAutoplay: true,        // Stop videos from auto-playing
  blockSocialWidgets: true,     // Block Facebook/Twitter embeds
  blockAnalytics: true,         // Block Google Analytics, Hotjar, etc.
  prefetchOptimize: true,       // Optimize DNS prefetching
  blockHeavyScripts: false,     // Block known heavy JS libraries
  reducedMotion: false,         // Reduce animations on websites
  textOnlyMode: false           // Extreme: text-only browsing mode
};

const DEFAULT_FILTER_LISTS = {
  easylist: true,
  easyprivacy: true,
  malware: true,
  annoyances: true,
  social: false
};

const DEFAULT_SPONSORBLOCK = {
  sponsor: true,
  intro: true,
  outro: true,
  interaction: true,
  selfpromo: false,
  music_offtopic: false,
  preview: false,
  filler: false
};

const DEFAULT_STATS = {
  adsBlocked: 0,
  trackersBlocked: 0,
  threatsBlocked: 0,
  dataSaved: 0,
  totalBlocked: 0,
  todayBlocked: 0,
  weekBlocked: 0,
  timeSaved: 0,
  lastReset: Date.now()
};

const DEFAULT_LICENSE = {
  tier: 'free', // 'free', 'premium', 'enterprise'
  key: null,
  status: 'inactive', // 'inactive', 'active', 'expired', 'revoked'
  expiresAt: null,
  activatedAt: null,
  lastValidated: null,
  features: {
    basicAdBlocking: true,
    popupBlocking: true,
    basicTrackerProtection: true,
    youtubeAdSkipping: false,
    sponsorblock: false,
    malwareProtection: false,
    fingerprintProtection: false,
    elementPicker: false,
    premiumThemes: false,
    policyManagement: false,
    ssoIntegration: false,
    auditLogging: false
  }
};

// Feature tier mapping
const FEATURE_TIERS = {
  free: ['basicAdBlocking', 'popupBlocking', 'basicTrackerProtection'],
  premium: ['basicAdBlocking', 'popupBlocking', 'basicTrackerProtection', 'youtubeAdSkipping', 'sponsorblock', 'malwareProtection', 'fingerprintProtection', 'elementPicker', 'premiumThemes'],
  enterprise: ['basicAdBlocking', 'popupBlocking', 'basicTrackerProtection', 'youtubeAdSkipping', 'sponsorblock', 'malwareProtection', 'fingerprintProtection', 'elementPicker', 'premiumThemes', 'policyManagement', 'ssoIntegration', 'auditLogging']
};

// ============================================
// UTILITY FUNCTIONS
// ============================================

/**
 * Format a number for display (1K, 1M, etc.)
 * @param {number} num - The number to format
 * @returns {string} Formatted number string
 */
function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1).replace(/\.0$/, '') + 'M';
  }
  if (num >= 1000) {
    return (num / 1000).toFixed(1).replace(/\.0$/, '') + 'K';
  }
  return num.toString();
}

/**
 * Format bytes to human readable format
 * @param {number} bytes - The bytes to format
 * @returns {string} Formatted size string
 */
function formatBytes(bytes) {
  if (bytes >= 1073741824) {
    return (bytes / 1073741824).toFixed(1) + ' GB';
  }
  if (bytes >= 1048576) {
    return (bytes / 1048576).toFixed(1) + ' MB';
  }
  if (bytes >= 1024) {
    return (bytes / 1024).toFixed(1) + ' KB';
  }
  return bytes + ' B';
}

/**
 * Format time in hours
 * @param {number} minutes - The minutes to format
 * @returns {string} Formatted time string
 */
function formatTime(minutes) {
  if (minutes >= 60) {
    return Math.floor(minutes / 60) + 'h ' + (minutes % 60) + 'm';
  }
  return minutes + 'm';
}

/**
 * Show a notification toast
 * @param {string} message - The message to display
 * @param {string} type - 'success' or 'error'
 */
function showNotification(message, type = 'success') {
  const notification = document.getElementById('notification');
  const notificationText = document.getElementById('notification-text');

  notificationText.textContent = message;
  notification.className = 'notification ' + type;
  notification.classList.add('show');

  setTimeout(() => {
    notification.classList.remove('show');
  }, 3000);
}

/**
 * Debounce function to limit rapid calls
 * @param {Function} func - The function to debounce
 * @param {number} wait - The delay in milliseconds
 * @returns {Function} Debounced function
 */
function debounce(func, wait) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

/**
 * Create an SVG element with the given path
 * @param {string} pathData - The SVG path data
 * @returns {SVGElement} The created SVG element
 */
function createSvgIcon(pathData) {
  const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  svg.setAttribute('width', '20');
  svg.setAttribute('height', '20');
  svg.setAttribute('viewBox', '0 0 24 24');
  svg.setAttribute('fill', 'none');
  svg.setAttribute('stroke', 'currentColor');
  svg.setAttribute('stroke-width', '2');

  const paths = pathData.split('|');
  paths.forEach(p => {
    if (p.startsWith('circle:')) {
      const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
      const params = p.replace('circle:', '').split(',');
      circle.setAttribute('cx', params[0]);
      circle.setAttribute('cy', params[1]);
      circle.setAttribute('r', params[2]);
      svg.appendChild(circle);
    } else {
      const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
      path.setAttribute('d', p);
      svg.appendChild(path);
    }
  });

  return svg;
}

// ============================================
// STORAGE FUNCTIONS
// ============================================

/**
 * Load settings from chrome.storage
 * @returns {Promise<Object>} The loaded settings
 */
async function loadSettings() {
  return new Promise((resolve) => {
    chrome.storage.sync.get([
      STORAGE_KEYS.CONFIG,
      STORAGE_KEYS.THEME,
      STORAGE_KEYS.RULES,
      STORAGE_KEYS.WHITELIST,
      STORAGE_KEYS.FILTER_LISTS,
      STORAGE_KEYS.SPONSORBLOCK
    ], (result) => {
      resolve({
        config: result[STORAGE_KEYS.CONFIG] || DEFAULT_CONFIG,
        theme: result[STORAGE_KEYS.THEME] || 'cyber-blue',
        rules: result[STORAGE_KEYS.RULES] || '',
        whitelist: result[STORAGE_KEYS.WHITELIST] || [],
        filterLists: result[STORAGE_KEYS.FILTER_LISTS] || DEFAULT_FILTER_LISTS,
        sponsorblock: result[STORAGE_KEYS.SPONSORBLOCK] || DEFAULT_SPONSORBLOCK
      });
    });
  });
}

/**
 * Load statistics from chrome.storage.local
 * @returns {Promise<Object>} The loaded statistics
 */
async function loadStats() {
  return new Promise((resolve) => {
    chrome.storage.local.get([STORAGE_KEYS.STATS], (result) => {
      resolve(result[STORAGE_KEYS.STATS] || DEFAULT_STATS);
    });
  });
}

/**
 * Save config to chrome.storage.sync
 * @param {Object} config - The config object to save
 */
async function saveConfig(config) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.CONFIG]: config }, () => {
      resolve();
    });
  });
}

/**
 * Save theme to chrome.storage.sync
 * @param {string} theme - The theme name to save
 */
async function saveTheme(theme) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.THEME]: theme }, () => {
      resolve();
    });
  });
}

/**
 * Save custom rules to chrome.storage.sync
 * @param {string} rules - The custom rules text
 */
async function saveRules(rules) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.RULES]: rules }, () => {
      resolve();
    });
  });
}

/**
 * Save whitelist to chrome.storage.sync
 * @param {Array} whitelist - The whitelist array
 */
async function saveWhitelist(whitelist) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.WHITELIST]: whitelist }, () => {
      resolve();
    });
  });
}

/**
 * Save filter lists to chrome.storage.sync
 * @param {Object} filterLists - The filter lists object
 */
async function saveFilterLists(filterLists) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.FILTER_LISTS]: filterLists }, () => {
      resolve();
    });
  });
}

/**
 * Save SponsorBlock categories to chrome.storage.sync
 * @param {Object} categories - The SponsorBlock categories object
 */
async function saveSponsorBlock(categories) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.SPONSORBLOCK]: categories }, () => {
      resolve();
    });
  });
}

/**
 * Save statistics to chrome.storage.local
 * @param {Object} stats - The statistics object
 */
async function saveStats(stats) {
  return new Promise((resolve) => {
    chrome.storage.local.set({ [STORAGE_KEYS.STATS]: stats }, () => {
      resolve();
    });
  });
}

/**
 * Load license from chrome.storage.sync
 * @returns {Promise<Object>} The loaded license
 */
async function loadLicense() {
  return new Promise((resolve) => {
    chrome.storage.sync.get([STORAGE_KEYS.LICENSE], (result) => {
      resolve(result[STORAGE_KEYS.LICENSE] || DEFAULT_LICENSE);
    });
  });
}

/**
 * Save license to chrome.storage.sync
 * @param {Object} license - The license object
 */
async function saveLicense(license) {
  return new Promise((resolve) => {
    chrome.storage.sync.set({ [STORAGE_KEYS.LICENSE]: license }, () => {
      resolve();
    });
  });
}

// ============================================
// DESKTOP APP DETECTION & LICENSE SYNC
// ============================================

/**
 * Desktop app connection state
 */
let desktopAppConnected = false;

/**
 * Detect if the desktop app is running
 * @returns {Promise<boolean>} Whether the desktop app is detected
 */
async function detectDesktopApp() {
  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), DESKTOP_APP_CONFIG.timeout);

    const response = await fetch(DESKTOP_APP_CONFIG.baseUrl + DESKTOP_APP_CONFIG.endpoints.health, {
      method: 'GET',
      signal: controller.signal,
      headers: {
        'Accept': 'application/json'
      }
    });

    clearTimeout(timeoutId);

    if (response.ok) {
      const data = await response.json();
      desktopAppConnected = true;
      return { connected: true, version: data.version || 'Unknown', status: data.status || 'online' };
    }

    desktopAppConnected = false;
    return { connected: false };
  } catch (error) {
    desktopAppConnected = false;
    return { connected: false, error: error.message };
  }
}

/**
 * Sync license from the desktop app
 * @returns {Promise<Object>} The synced license data or error
 */
async function syncLicenseFromDesktop() {
  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), DESKTOP_APP_CONFIG.timeout * 2);

    const response = await fetch(DESKTOP_APP_CONFIG.baseUrl + DESKTOP_APP_CONFIG.endpoints.license, {
      method: 'GET',
      signal: controller.signal,
      headers: {
        'Accept': 'application/json'
      }
    });

    clearTimeout(timeoutId);

    if (!response.ok) {
      throw new Error('Failed to fetch license from desktop app');
    }

    const data = await response.json();

    // Build license object from desktop app response
    const license = {
      tier: data.tier || 'free',
      key: data.license_key || null,
      status: data.status || 'inactive',
      expiresAt: data.expires_at || null,
      activatedAt: data.activated_at || null,
      lastValidated: Date.now(),
      features: buildFeatureSet(data.tier || 'free')
    };

    // Save the synced license
    await saveLicense(license);

    return { success: true, license };
  } catch (error) {
    console.error('License sync error:', error);
    return { success: false, error: error.message };
  }
}

/**
 * Activate a license key via the desktop app or directly
 * @param {string} licenseKey - The license key to activate
 * @returns {Promise<Object>} The activation result
 */
async function activateLicenseKey(licenseKey) {
  // First, try via desktop app if connected
  if (desktopAppConnected) {
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), DESKTOP_APP_CONFIG.timeout * 2);

      const response = await fetch(DESKTOP_APP_CONFIG.baseUrl + DESKTOP_APP_CONFIG.endpoints.licenseActivate, {
        method: 'POST',
        signal: controller.signal,
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ license_key: licenseKey })
      });

      clearTimeout(timeoutId);

      if (response.ok) {
        const data = await response.json();
        const license = {
          tier: data.tier || 'premium',
          key: licenseKey,
          status: 'active',
          expiresAt: data.expires_at || null,
          activatedAt: Date.now(),
          lastValidated: Date.now(),
          features: buildFeatureSet(data.tier || 'premium')
        };

        await saveLicense(license);
        return { success: true, license, message: 'License activated via desktop app' };
      }

      const errorData = await response.json();
      return { success: false, error: errorData.message || 'Activation failed' };
    } catch (error) {
      console.log('Desktop activation failed, trying direct activation:', error.message);
    }
  }

  // Fallback: validate license key format and store locally
  // In production, this would call a remote license server
  if (!isValidLicenseKeyFormat(licenseKey)) {
    return { success: false, error: 'Invalid license key format' };
  }

  // For demo/development: accept any valid format key as premium
  // In production, this would validate against a license server
  const license = {
    tier: 'premium',
    key: licenseKey,
    status: 'active',
    expiresAt: new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString(), // 1 year
    activatedAt: Date.now(),
    lastValidated: Date.now(),
    features: buildFeatureSet('premium')
  };

  await saveLicense(license);
  return { success: true, license, message: 'License activated' };
}

/**
 * Deactivate the current license
 * @returns {Promise<Object>} The deactivation result
 */
async function deactivateLicense() {
  const license = { ...DEFAULT_LICENSE };
  await saveLicense(license);

  // Notify background script
  chrome.runtime.sendMessage({
    action: 'licenseUpdated',
    license: license
  });

  return { success: true };
}

/**
 * Validate license key format
 * @param {string} key - The license key to validate
 * @returns {boolean} Whether the key format is valid
 */
function isValidLicenseKeyFormat(key) {
  // Format: XXXX-XXXX-XXXX-XXXX (alphanumeric)
  const pattern = /^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
  return pattern.test(key.toUpperCase());
}

/**
 * Build feature set based on tier
 * @param {string} tier - The license tier
 * @returns {Object} The feature set
 */
function buildFeatureSet(tier) {
  const features = { ...DEFAULT_LICENSE.features };
  const tierFeatures = FEATURE_TIERS[tier] || FEATURE_TIERS.free;

  for (const feature of tierFeatures) {
    features[feature] = true;
  }

  return features;
}

/**
 * Format license key for display (mask middle segments)
 * @param {string} key - The license key
 * @returns {string} The masked key
 */
function maskLicenseKey(key) {
  if (!key) return '****-****-****-****';
  const parts = key.split('-');
  if (parts.length !== 4) return key;
  return parts[0] + '-****-****-' + parts[3];
}

/**
 * Format expiry date for display
 * @param {string|number} expiresAt - The expiry timestamp or ISO string
 * @returns {string} Formatted expiry string
 */
function formatExpiryDate(expiresAt) {
  if (!expiresAt) return 'Never';

  const date = new Date(expiresAt);
  const now = new Date();

  if (date < now) {
    return 'Expired';
  }

  const diffDays = Math.ceil((date - now) / (1000 * 60 * 60 * 24));

  if (diffDays <= 7) {
    return `Expires in ${diffDays} day${diffDays === 1 ? '' : 's'}`;
  }

  return date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
}

// ============================================
// UI UPDATE FUNCTIONS
// ============================================

/**
 * Apply theme to the page
 * @param {string} themeName - The theme name to apply
 */
function applyTheme(themeName) {
  document.documentElement.setAttribute('data-theme', themeName);

  // Update active state on theme options
  document.querySelectorAll('.theme-option').forEach(option => {
    option.classList.remove('active');
    if (option.dataset.theme === themeName) {
      option.classList.add('active');
    }
  });
}

/**
 * Update toggle states from config
 * @param {Object} config - The config object
 */
function updateToggles(config) {
  // Map config keys to toggle element IDs
  const toggleMap = {
    protection: 'toggle-protection',
    blockAds: 'toggle-ads',
    blockPopups: 'toggle-popups',
    blockRedirects: 'toggle-redirects',
    showBadge: 'toggle-badge',
    threatAlerts: 'toggle-threat-alerts',
    animations: 'toggle-animations',
    compactMode: 'toggle-compact',
    youtubeAds: 'toggle-yt-ads',
    youtubeShorts: 'toggle-yt-shorts',
    youtubeDistractionFree: 'toggle-yt-distraction',
    sponsorblock: 'toggle-sponsorblock',
    malwareProtection: 'toggle-malware',
    phishingProtection: 'toggle-phishing',
    cryptominerBlocking: 'toggle-cryptominer',
    scamProtection: 'toggle-scam',
    blockTrackers: 'toggle-trackers',
    fingerprintProtection: 'toggle-fingerprint',
    webrtcLeak: 'toggle-webrtc',
    autoRejectCookies: 'toggle-cookies',
    hideCookieNotices: 'toggle-cookie-notices'
  };

  for (const [configKey, toggleId] of Object.entries(toggleMap)) {
    const toggle = document.getElementById(toggleId);
    if (toggle && config[configKey] !== undefined) {
      toggle.checked = config[configKey];
    }
  }
}

/**
 * Update filter list toggles
 * @param {Object} filterLists - The filter lists object
 */
function updateFilterLists(filterLists) {
  document.querySelectorAll('[data-filter]').forEach(toggle => {
    const filterName = toggle.dataset.filter;
    if (filterLists[filterName] !== undefined) {
      toggle.checked = filterLists[filterName];
    }
  });
}

/**
 * Update SponsorBlock segment toggles
 * @param {Object} categories - The SponsorBlock categories object
 */
function updateSponsorBlock(categories) {
  document.querySelectorAll('[data-segment]').forEach(toggle => {
    const segmentName = toggle.dataset.segment;
    if (categories[segmentName] !== undefined) {
      toggle.checked = categories[segmentName];
    }
  });
}

/**
 * Update statistics display
 * @param {Object} stats - The statistics object
 */
function updateStatsDisplay(stats) {
  // General page stats
  document.getElementById('stat-ads').textContent = formatNumber(stats.adsBlocked || 0);
  document.getElementById('stat-trackers').textContent = formatNumber(stats.trackersBlocked || 0);
  document.getElementById('stat-malware').textContent = formatNumber(stats.threatsBlocked || 0);
  document.getElementById('stat-data').textContent = formatBytes(stats.dataSaved || 0);

  // Statistics page
  document.getElementById('total-blocked').textContent = formatNumber(stats.totalBlocked || 0);
  document.getElementById('today-blocked').textContent = formatNumber(stats.todayBlocked || 0);
  document.getElementById('week-blocked').textContent = formatNumber(stats.weekBlocked || 0);
  document.getElementById('time-saved').textContent = formatTime(stats.timeSaved || 0);

  // Breakdown
  document.getElementById('breakdown-ads').textContent = formatNumber(stats.adsBlocked || 0);
  document.getElementById('breakdown-trackers').textContent = formatNumber(stats.trackersBlocked || 0);
  document.getElementById('breakdown-threats').textContent = formatNumber(stats.threatsBlocked || 0);
}

/**
 * Create a status dot element with span text using safe DOM methods
 * @param {string} statusClass - The status class (active, expired, inactive)
 * @param {string} text - The status text
 * @returns {DocumentFragment} The created elements
 */
function createStatusElement(statusClass, text) {
  const fragment = document.createDocumentFragment();
  const dot = document.createElement('span');
  dot.className = 'status-dot ' + statusClass;
  fragment.appendChild(dot);
  fragment.appendChild(document.createTextNode(' ' + text));
  return fragment;
}

/**
 * Create an SVG icon element using safe DOM methods
 * @param {string} type - The icon type ('checkmark' or 'lock')
 * @returns {SVGElement} The created SVG element
 */
function createFeatureIcon(type) {
  const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  svg.setAttribute('viewBox', '0 0 24 24');
  svg.setAttribute('width', '20');
  svg.setAttribute('height', '20');
  svg.setAttribute('fill', 'none');
  svg.setAttribute('stroke', 'currentColor');
  svg.setAttribute('stroke-width', '2');

  if (type === 'checkmark') {
    const polyline = document.createElementNS('http://www.w3.org/2000/svg', 'polyline');
    polyline.setAttribute('points', '20 6 9 17 4 12');
    svg.appendChild(polyline);
  } else {
    // Lock icon
    const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    rect.setAttribute('x', '3');
    rect.setAttribute('y', '11');
    rect.setAttribute('width', '18');
    rect.setAttribute('height', '11');
    rect.setAttribute('rx', '2');
    rect.setAttribute('ry', '2');
    svg.appendChild(rect);

    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    path.setAttribute('d', 'M7 11V7a5 5 0 0 1 10 0v4');
    svg.appendChild(path);
  }

  return svg;
}

/**
 * Update license display
 * @param {Object} license - The license object
 */
function updateLicenseDisplay(license) {
  const tierNames = {
    free: 'Free Edition',
    premium: 'Premium Edition',
    enterprise: 'Enterprise Edition'
  };

  const tierDescs = {
    free: 'Basic ad blocking protection',
    premium: 'Full protection with all premium features',
    enterprise: 'Enterprise-grade protection with policy management'
  };

  // Update tier display
  const tierName = document.getElementById('license-tier-name');
  const tierDesc = document.getElementById('license-tier-desc');
  const licenseIcon = document.getElementById('license-icon');
  const licenseBadge = document.getElementById('license-badge');
  const navBadge = document.getElementById('nav-license-badge');

  if (tierName) tierName.textContent = tierNames[license.tier] || tierNames.free;
  if (tierDesc) tierDesc.textContent = tierDescs[license.tier] || tierDescs.free;

  // Update icon and badge based on tier
  if (licenseIcon) {
    licenseIcon.className = 'license-icon';
    if (license.tier === 'premium') {
      licenseIcon.classList.add('premium');
    } else if (license.tier === 'enterprise') {
      licenseIcon.classList.add('enterprise');
    }
  }

  if (licenseBadge) {
    licenseBadge.className = 'license-badge ' + license.tier;
    licenseBadge.textContent = license.tier.toUpperCase();
  }

  // Show/hide nav badge
  if (navBadge) {
    if (license.tier !== 'free' && license.status === 'active') {
      navBadge.style.display = 'inline-block';
      navBadge.textContent = license.tier === 'enterprise' ? 'ENT' : 'PRO';
    } else {
      navBadge.style.display = 'none';
    }
  }

  // Update license details section
  const licenseDetails = document.getElementById('license-details');
  const licenseKeyDisplay = document.getElementById('license-key-display');
  const licenseStatusDisplay = document.getElementById('license-status-display');
  const licenseExpiryDisplay = document.getElementById('license-expiry-display');

  if (license.tier !== 'free' && license.key) {
    if (licenseDetails) licenseDetails.style.display = 'block';
    if (licenseKeyDisplay) licenseKeyDisplay.textContent = maskLicenseKey(license.key);

    // Update status display using safe DOM methods
    if (licenseStatusDisplay) {
      // Clear existing content
      while (licenseStatusDisplay.firstChild) {
        licenseStatusDisplay.removeChild(licenseStatusDisplay.firstChild);
      }

      let statusClass = 'inactive';
      let statusText = 'Inactive';

      if (license.status === 'active') {
        statusClass = 'active';
        statusText = 'Active';
      } else if (license.status === 'expired') {
        statusClass = 'expired';
        statusText = 'Expired';
      }

      licenseStatusDisplay.appendChild(createStatusElement(statusClass, statusText));
    }

    if (licenseExpiryDisplay) {
      licenseExpiryDisplay.textContent = formatExpiryDate(license.expiresAt);
    }
  } else {
    if (licenseDetails) licenseDetails.style.display = 'none';
  }

  // Show/hide upgrade CTA and license management
  const upgradeCta = document.getElementById('upgrade-cta-card');
  const licenseManagement = document.getElementById('license-management-card');

  if (license.tier !== 'free' && license.status === 'active') {
    if (upgradeCta) upgradeCta.style.display = 'none';
    if (licenseManagement) licenseManagement.style.display = 'block';
  } else {
    if (upgradeCta) upgradeCta.style.display = 'block';
    if (licenseManagement) licenseManagement.style.display = 'none';
  }

  // Update feature items
  updateFeatureDisplay(license);
}

/**
 * Update feature display based on license
 * @param {Object} license - The license object
 */
function updateFeatureDisplay(license) {
  const featureMapping = {
    'feature-youtube-ads': 'youtubeAdSkipping',
    'feature-sponsorblock': 'sponsorblock',
    'feature-malware': 'malwareProtection',
    'feature-fingerprint': 'fingerprintProtection',
    'feature-element-picker': 'elementPicker',
    'feature-themes': 'premiumThemes',
    'feature-policy': 'policyManagement',
    'feature-sso': 'ssoIntegration',
    'feature-audit': 'auditLogging'
  };

  for (const [elementId, featureKey] of Object.entries(featureMapping)) {
    const element = document.getElementById(elementId);
    if (!element) continue;

    const isUnlocked = license.features && license.features[featureKey];
    const featureIcon = element.querySelector('.feature-icon');

    if (isUnlocked) {
      element.classList.remove('locked');
      element.classList.add('unlocked');

      if (featureIcon) {
        featureIcon.classList.remove('locked');
        featureIcon.classList.add('unlocked');
        // Clear and add checkmark icon
        while (featureIcon.firstChild) {
          featureIcon.removeChild(featureIcon.firstChild);
        }
        featureIcon.appendChild(createFeatureIcon('checkmark'));
      }
    } else {
      element.classList.remove('unlocked');
      element.classList.add('locked');

      if (featureIcon) {
        featureIcon.classList.remove('unlocked');
        featureIcon.classList.add('locked');
        // Clear and add lock icon
        while (featureIcon.firstChild) {
          featureIcon.removeChild(featureIcon.firstChild);
        }
        const isEnterprise = element.dataset.tier === 'enterprise';
        if (isEnterprise) featureIcon.classList.add('enterprise');
        featureIcon.appendChild(createFeatureIcon('lock'));
      }
    }
  }
}

/**
 * Update desktop app status display
 * @param {Object} status - The desktop app status
 */
function updateDesktopAppStatus(status) {
  const statusIndicator = document.getElementById('desktop-status-indicator');
  const statusText = document.getElementById('desktop-status-text');
  const statusDot = statusIndicator ? statusIndicator.querySelector('.status-dot') : null;

  if (!statusIndicator || !statusText) return;

  if (status.connected) {
    if (statusDot) {
      statusDot.className = 'status-dot connected';
    }
    statusText.textContent = 'Desktop App Connected (v' + (status.version || 'Unknown') + ')';
    statusText.style.color = 'var(--success)';
  } else {
    if (statusDot) {
      statusDot.className = 'status-dot disconnected';
    }
    statusText.textContent = 'Desktop App Not Detected';
    statusText.style.color = 'var(--text-muted)';
  }
}

/**
 * Create a whitelist item element using safe DOM methods
 * @param {string} domain - The domain name
 * @returns {HTMLElement} The created filter item element
 */
function createWhitelistItem(domain) {
  // Create the main container
  const filterItem = document.createElement('div');
  filterItem.className = 'filter-item';

  // Create the icon container
  const filterIcon = document.createElement('div');
  filterIcon.className = 'filter-icon';
  filterIcon.style.background = 'rgba(var(--primary-rgb), 0.15)';

  // Create SVG icon for globe
  const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  svg.setAttribute('width', '20');
  svg.setAttribute('height', '20');
  svg.setAttribute('viewBox', '0 0 24 24');
  svg.setAttribute('fill', 'none');
  svg.setAttribute('stroke', 'currentColor');
  svg.setAttribute('stroke-width', '2');

  const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
  circle.setAttribute('cx', '12');
  circle.setAttribute('cy', '12');
  circle.setAttribute('r', '10');
  svg.appendChild(circle);

  const path1 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
  path1.setAttribute('d', 'M2 12h20');
  svg.appendChild(path1);

  const path2 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
  path2.setAttribute('d', 'M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z');
  svg.appendChild(path2);

  filterIcon.appendChild(svg);

  // Create the info container
  const filterInfo = document.createElement('div');
  filterInfo.className = 'filter-info';

  const filterName = document.createElement('div');
  filterName.className = 'filter-name';
  filterName.textContent = domain; // Safe: using textContent

  const filterMeta = document.createElement('div');
  filterMeta.className = 'filter-meta';
  filterMeta.textContent = 'Protection disabled';

  filterInfo.appendChild(filterName);
  filterInfo.appendChild(filterMeta);

  // Create the remove button
  const removeBtn = document.createElement('button');
  removeBtn.className = 'btn btn-danger';
  removeBtn.style.padding = '8px 16px';
  removeBtn.textContent = 'Remove';
  removeBtn.addEventListener('click', async () => {
    await removeDomainFromWhitelist(domain);
  });

  // Assemble the item
  filterItem.appendChild(filterIcon);
  filterItem.appendChild(filterInfo);
  filterItem.appendChild(removeBtn);

  return filterItem;
}

/**
 * Update whitelist display using safe DOM methods
 * @param {Array} whitelist - The whitelist array
 */
function updateWhitelistDisplay(whitelist) {
  const container = document.getElementById('whitelist-items');

  // Clear existing content safely
  while (container.firstChild) {
    container.removeChild(container.firstChild);
  }

  if (!whitelist || whitelist.length === 0) {
    const emptyMessage = document.createElement('p');
    emptyMessage.style.cssText = 'color: var(--text-muted); text-align: center; padding: 24px;';
    emptyMessage.textContent = 'No whitelisted domains';
    container.appendChild(emptyMessage);
    return;
  }

  // Create and append each whitelist item using safe DOM methods
  whitelist.forEach(domain => {
    const item = createWhitelistItem(domain);
    container.appendChild(item);
  });
}

// ============================================
// WHITELIST FUNCTIONS
// ============================================

/**
 * Add a domain to the whitelist
 * @param {string} domain - The domain to add
 */
async function addDomainToWhitelist(domain) {
  // Clean up the domain
  domain = domain.trim().toLowerCase();

  // Remove protocol if present
  domain = domain.replace(/^https?:\/\//, '');

  // Remove trailing slash and path
  domain = domain.split('/')[0];

  // Remove www. prefix
  domain = domain.replace(/^www\./, '');

  if (!domain) {
    showNotification('Please enter a valid domain', 'error');
    return;
  }

  // Basic domain validation
  const domainPattern = /^[a-z0-9]+([\-\.]{1}[a-z0-9]+)*\.[a-z]{2,}$/;
  if (!domainPattern.test(domain)) {
    showNotification('Invalid domain format', 'error');
    return;
  }

  const settings = await loadSettings();
  const whitelist = settings.whitelist || [];

  if (whitelist.includes(domain)) {
    showNotification('Domain already whitelisted', 'error');
    return;
  }

  whitelist.push(domain);
  await saveWhitelist(whitelist);
  updateWhitelistDisplay(whitelist);

  // Clear input
  document.getElementById('whitelist-domain').value = '';

  showNotification(domain + ' added to whitelist');

  // Notify background script
  chrome.runtime.sendMessage({
    action: 'whitelistUpdated',
    whitelist: whitelist
  });
}

/**
 * Remove a domain from the whitelist
 * @param {string} domain - The domain to remove
 */
async function removeDomainFromWhitelist(domain) {
  const settings = await loadSettings();
  const whitelist = (settings.whitelist || []).filter(d => d !== domain);

  await saveWhitelist(whitelist);
  updateWhitelistDisplay(whitelist);

  showNotification(domain + ' removed from whitelist');

  // Notify background script
  chrome.runtime.sendMessage({
    action: 'whitelistUpdated',
    whitelist: whitelist
  });
}

// ============================================
// IMPORT/EXPORT FUNCTIONS
// ============================================

/**
 * Export all settings to a JSON file
 */
async function exportSettings() {
  const settings = await loadSettings();
  const stats = await loadStats();

  const exportData = {
    version: '4.0.0',
    exportDate: new Date().toISOString(),
    settings: {
      config: settings.config,
      theme: settings.theme,
      rules: settings.rules,
      whitelist: settings.whitelist,
      filterLists: settings.filterLists,
      sponsorblock: settings.sponsorblock
    },
    stats: stats
  };

  const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
  const url = URL.createObjectURL(blob);

  const a = document.createElement('a');
  a.href = url;
  a.download = 'streamshield-backup-' + new Date().toISOString().split('T')[0] + '.json';
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  URL.revokeObjectURL(url);

  showNotification('Settings exported successfully');
}

/**
 * Import settings from a JSON file
 * @param {File} file - The file to import
 */
async function importSettings(file) {
  try {
    const text = await file.text();
    const data = JSON.parse(text);

    // Validate the import data
    if (!data.settings) {
      throw new Error('Invalid backup file format');
    }

    // Import settings
    if (data.settings.config) {
      await saveConfig({ ...DEFAULT_CONFIG, ...data.settings.config });
    }
    if (data.settings.theme) {
      await saveTheme(data.settings.theme);
    }
    if (data.settings.rules) {
      await saveRules(data.settings.rules);
    }
    if (data.settings.whitelist) {
      await saveWhitelist(data.settings.whitelist);
    }
    if (data.settings.filterLists) {
      await saveFilterLists({ ...DEFAULT_FILTER_LISTS, ...data.settings.filterLists });
    }
    if (data.settings.sponsorblock) {
      await saveSponsorBlock({ ...DEFAULT_SPONSORBLOCK, ...data.settings.sponsorblock });
    }

    // Optionally import stats
    if (data.stats) {
      await saveStats({ ...DEFAULT_STATS, ...data.stats });
    }

    showNotification('Settings imported successfully. Reloading...');

    // Reload the page to apply changes
    setTimeout(() => {
      window.location.reload();
    }, 1500);

  } catch (error) {
    console.error('Import error:', error);
    showNotification('Failed to import settings: ' + error.message, 'error');
  }
}

/**
 * Reset all settings to defaults
 */
async function resetSettings() {
  if (!confirm('Are you sure you want to reset all settings to defaults? This cannot be undone.')) {
    return;
  }

  await saveConfig(DEFAULT_CONFIG);
  await saveTheme('cyber-blue');
  await saveRules('');
  await saveWhitelist([]);
  await saveFilterLists(DEFAULT_FILTER_LISTS);
  await saveSponsorBlock(DEFAULT_SPONSORBLOCK);

  showNotification('Settings reset to defaults. Reloading...');

  setTimeout(() => {
    window.location.reload();
  }, 1500);
}

/**
 * Reset statistics
 */
async function resetStats() {
  if (!confirm('Are you sure you want to reset all statistics?')) {
    return;
  }

  const stats = { ...DEFAULT_STATS, lastReset: Date.now() };
  await saveStats(stats);
  updateStatsDisplay(stats);

  showNotification('Statistics reset successfully');

  // Notify background script
  chrome.runtime.sendMessage({ action: 'statsReset' });
}

// ============================================
// NAVIGATION FUNCTIONS
// ============================================

/**
 * Switch to a page section
 * @param {string} pageName - The page name to switch to
 */
function switchPage(pageName) {
  // Update nav items
  document.querySelectorAll('.nav-item').forEach(item => {
    item.classList.remove('active');
    if (item.dataset.page === pageName) {
      item.classList.add('active');
    }
  });

  // Update page sections with smooth transition
  document.querySelectorAll('.page-section').forEach(section => {
    if (section.id === 'page-' + pageName) {
      section.style.opacity = '0';
      section.classList.add('active');
      requestAnimationFrame(() => {
        section.style.transition = 'opacity 0.3s ease';
        section.style.opacity = '1';
      });
    } else {
      section.classList.remove('active');
    }
  });

  // Scroll to top of main content
  document.querySelector('.main-content').scrollTop = 0;
}

// ============================================
// ELEMENT PICKER FUNCTION
// ============================================

/**
 * Launch the element picker tool
 */
function launchElementPicker() {
  // Get the current active tab
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs.length > 0) {
      // Send message to background script to activate element picker
      chrome.runtime.sendMessage({
        action: 'launchElementPicker',
        tabId: tabs[0].id
      }, (response) => {
        if (response && response.success) {
          showNotification('Element picker activated. Click on any element to block it.');
          // Close the options page after a short delay
          setTimeout(() => {
            window.close();
          }, 1000);
        } else {
          showNotification('Could not activate element picker. Try on a different tab.', 'error');
        }
      });
    }
  });
}

// ============================================
// VALIDATION FUNCTIONS
// ============================================

/**
 * Validate custom filter rules syntax
 * @param {string} rules - The rules text to validate
 * @returns {Object} Validation result with valid boolean and errors array
 */
function validateRulesSyntax(rules) {
  const lines = rules.split('\n');
  const errors = [];

  lines.forEach((line, index) => {
    const lineNum = index + 1;
    const trimmedLine = line.trim();

    // Skip empty lines and comments
    if (!trimmedLine || trimmedLine.startsWith('!')) {
      return;
    }

    // Basic syntax checks
    if (trimmedLine.includes('##') || trimmedLine.includes('#@#')) {
      // Element hiding rule - basic check
      const parts = trimmedLine.split(/##|#@#/);
      if (parts.length < 2 || !parts[1].trim()) {
        errors.push('Line ' + lineNum + ': Invalid element hiding rule - missing selector');
      }
    } else if (trimmedLine.startsWith('@@')) {
      // Exception rule - should have valid pattern
      const pattern = trimmedLine.substring(2);
      if (!pattern) {
        errors.push('Line ' + lineNum + ': Empty exception rule');
      }
    } else if (trimmedLine.startsWith('||')) {
      // Domain-anchored rule - basic check
      if (trimmedLine.length < 4) {
        errors.push('Line ' + lineNum + ': Incomplete domain rule');
      }
    } else if (trimmedLine.startsWith('|') && !trimmedLine.startsWith('||')) {
      // Address start anchor
      if (trimmedLine.length < 2) {
        errors.push('Line ' + lineNum + ': Incomplete anchored rule');
      }
    }

    // Check for unbalanced brackets in options
    if (trimmedLine.includes('$')) {
      const optionPart = trimmedLine.split('$')[1];
      if (optionPart) {
        const openBrackets = (optionPart.match(/\[/g) || []).length;
        const closeBrackets = (optionPart.match(/\]/g) || []).length;
        if (openBrackets !== closeBrackets) {
          errors.push('Line ' + lineNum + ': Unbalanced brackets in options');
        }
      }
    }
  });

  return {
    valid: errors.length === 0,
    errors: errors
  };
}

// ============================================
// EVENT HANDLERS SETUP
// ============================================

/**
 * Initialize all event handlers
 */
function setupEventHandlers() {
  // Navigation clicks
  document.querySelectorAll('.nav-item').forEach(item => {
    item.addEventListener('click', () => {
      const pageName = item.dataset.page;
      if (pageName) {
        switchPage(pageName);
      }
    });
  });

  // Theme selection
  document.querySelectorAll('.theme-option').forEach(option => {
    option.addEventListener('click', async () => {
      const themeName = option.dataset.theme;
      applyTheme(themeName);
      await saveTheme(themeName);
      showNotification('Theme updated');

      // Notify other extension pages
      chrome.runtime.sendMessage({ action: 'themeChanged', theme: themeName });
    });
  });

  // Config toggles
  const configToggles = [
    'toggle-protection', 'toggle-ads', 'toggle-popups', 'toggle-redirects',
    'toggle-badge', 'toggle-threat-alerts', 'toggle-animations', 'toggle-compact',
    'toggle-yt-ads', 'toggle-yt-shorts', 'toggle-yt-distraction', 'toggle-sponsorblock',
    'toggle-malware', 'toggle-phishing', 'toggle-cryptominer', 'toggle-scam',
    'toggle-trackers', 'toggle-fingerprint', 'toggle-webrtc',
    'toggle-cookies', 'toggle-cookie-notices'
  ];

  const toggleToConfigKey = {
    'toggle-protection': 'protection',
    'toggle-ads': 'blockAds',
    'toggle-popups': 'blockPopups',
    'toggle-redirects': 'blockRedirects',
    'toggle-badge': 'showBadge',
    'toggle-threat-alerts': 'threatAlerts',
    'toggle-animations': 'animations',
    'toggle-compact': 'compactMode',
    'toggle-yt-ads': 'youtubeAds',
    'toggle-yt-shorts': 'youtubeShorts',
    'toggle-yt-distraction': 'youtubeDistractionFree',
    'toggle-sponsorblock': 'sponsorblock',
    'toggle-malware': 'malwareProtection',
    'toggle-phishing': 'phishingProtection',
    'toggle-cryptominer': 'cryptominerBlocking',
    'toggle-scam': 'scamProtection',
    'toggle-trackers': 'blockTrackers',
    'toggle-fingerprint': 'fingerprintProtection',
    'toggle-webrtc': 'webrtcLeak',
    'toggle-cookies': 'autoRejectCookies',
    'toggle-cookie-notices': 'hideCookieNotices'
  };

  configToggles.forEach(toggleId => {
    const toggle = document.getElementById(toggleId);
    if (toggle) {
      toggle.addEventListener('change', debounce(async () => {
        const settings = await loadSettings();
        const configKey = toggleToConfigKey[toggleId];
        if (configKey) {
          settings.config[configKey] = toggle.checked;
          await saveConfig(settings.config);
          showNotification('Settings saved');

          // Notify background script
          chrome.runtime.sendMessage({
            action: 'configUpdated',
            config: settings.config
          });
        }
      }, 300));
    }
  });

  // Filter list toggles
  document.querySelectorAll('[data-filter]').forEach(toggle => {
    toggle.addEventListener('change', debounce(async () => {
      const settings = await loadSettings();
      const filterName = toggle.dataset.filter;
      settings.filterLists[filterName] = toggle.checked;
      await saveFilterLists(settings.filterLists);
      showNotification('Filter list updated');

      // Notify background script
      chrome.runtime.sendMessage({
        action: 'filterListsUpdated',
        filterLists: settings.filterLists
      });
    }, 300));
  });

  // SponsorBlock segment toggles
  document.querySelectorAll('[data-segment]').forEach(toggle => {
    toggle.addEventListener('change', debounce(async () => {
      const settings = await loadSettings();
      const segmentName = toggle.dataset.segment;
      settings.sponsorblock[segmentName] = toggle.checked;
      await saveSponsorBlock(settings.sponsorblock);
      showNotification('SponsorBlock settings saved');

      // Notify background script
      chrome.runtime.sendMessage({
        action: 'sponsorblockUpdated',
        categories: settings.sponsorblock
      });
    }, 300));
  });

  // Save custom rules button
  const saveRulesBtn = document.getElementById('btn-save-rules');
  if (saveRulesBtn) {
    saveRulesBtn.addEventListener('click', async () => {
      const rulesTextarea = document.getElementById('custom-rules');
      const rules = rulesTextarea.value;
      await saveRules(rules);
      showNotification('Custom rules saved');

      // Notify background script
      chrome.runtime.sendMessage({
        action: 'rulesUpdated',
        rules: rules
      });
    });
  }

  // Validate rules button
  const validateRulesBtn = document.getElementById('btn-validate-rules');
  if (validateRulesBtn) {
    validateRulesBtn.addEventListener('click', () => {
      const rulesTextarea = document.getElementById('custom-rules');
      const result = validateRulesSyntax(rulesTextarea.value);

      if (result.valid) {
        showNotification('All rules are valid!');
      } else {
        showNotification('Found ' + result.errors.length + ' error(s): ' + result.errors[0], 'error');
        console.log('Validation errors:', result.errors);
      }
    });
  }

  // Element picker button
  const elementPickerBtn = document.getElementById('btn-element-picker');
  if (elementPickerBtn) {
    elementPickerBtn.addEventListener('click', launchElementPicker);
  }

  // Whitelist add button
  const addWhitelistBtn = document.getElementById('btn-add-whitelist');
  if (addWhitelistBtn) {
    addWhitelistBtn.addEventListener('click', () => {
      const input = document.getElementById('whitelist-domain');
      addDomainToWhitelist(input.value);
    });
  }

  // Whitelist input enter key
  const whitelistInput = document.getElementById('whitelist-domain');
  if (whitelistInput) {
    whitelistInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        addDomainToWhitelist(whitelistInput.value);
      }
    });
  }

  // Export button
  const exportBtn = document.getElementById('btn-export');
  if (exportBtn) {
    exportBtn.addEventListener('click', exportSettings);
  }

  // Import button
  const importBtn = document.getElementById('btn-import');
  if (importBtn) {
    importBtn.addEventListener('click', () => {
      document.getElementById('import-file').click();
    });
  }

  // Import file change
  const importFile = document.getElementById('import-file');
  if (importFile) {
    importFile.addEventListener('change', (e) => {
      const file = e.target.files[0];
      if (file) {
        importSettings(file);
        e.target.value = ''; // Reset file input
      }
    });
  }

  // Reset settings button
  const resetSettingsBtn = document.getElementById('btn-reset-settings');
  if (resetSettingsBtn) {
    resetSettingsBtn.addEventListener('click', resetSettings);
  }

  // Reset stats button
  const resetStatsBtn = document.getElementById('btn-reset-stats');
  if (resetStatsBtn) {
    resetStatsBtn.addEventListener('click', resetStats);
  }

  // Update filters button
  const updateFiltersBtn = document.getElementById('btn-update-filters');
  if (updateFiltersBtn) {
    updateFiltersBtn.addEventListener('click', () => {
      chrome.runtime.sendMessage({ action: 'updateFilters' }, (response) => {
        if (response && response.success) {
          showNotification('Filter lists updated');
        } else {
          showNotification('Failed to update filter lists', 'error');
        }
      });
    });
  }

  // Add custom filter list button
  const addFilterBtn = document.getElementById('btn-add-filter');
  if (addFilterBtn) {
    addFilterBtn.addEventListener('click', () => {
      const input = document.getElementById('custom-filter-url');
      const url = input.value.trim();

      if (!url) {
        showNotification('Please enter a filter list URL', 'error');
        return;
      }

      // Basic URL validation
      try {
        new URL(url);
      } catch (e) {
        showNotification('Invalid URL format', 'error');
        return;
      }

      chrome.runtime.sendMessage({
        action: 'addCustomFilterList',
        url: url
      }, (response) => {
        if (response && response.success) {
          showNotification('Filter list added');
          input.value = '';
        } else {
          showNotification('Failed to add filter list', 'error');
        }
      });
    });
  }

  // ============================================
  // LICENSE EVENT HANDLERS
  // ============================================

  // Sync with Desktop App button
  const syncDesktopBtn = document.getElementById('btn-sync-desktop');
  if (syncDesktopBtn) {
    syncDesktopBtn.addEventListener('click', async () => {
      // Show loading state
      syncDesktopBtn.classList.add('btn-loading');
      syncDesktopBtn.disabled = true;

      // First check if desktop app is available
      const status = await detectDesktopApp();
      updateDesktopAppStatus(status);

      if (!status.connected) {
        showNotification('Desktop app not detected. Please ensure StreamShield is running.', 'error');
        syncDesktopBtn.classList.remove('btn-loading');
        syncDesktopBtn.disabled = false;
        return;
      }

      // Try to sync license
      const result = await syncLicenseFromDesktop();

      syncDesktopBtn.classList.remove('btn-loading');
      syncDesktopBtn.disabled = false;

      if (result.success) {
        showNotification('License synced successfully!');
        updateLicenseDisplay(result.license);

        // Notify background script
        chrome.runtime.sendMessage({
          action: 'licenseUpdated',
          license: result.license
        });
      } else {
        showNotification('Failed to sync license: ' + (result.error || 'Unknown error'), 'error');
      }
    });
  }

  // Activate License button
  const activateLicenseBtn = document.getElementById('btn-activate-license');
  const licenseKeyInput = document.getElementById('license-key-input');

  if (activateLicenseBtn && licenseKeyInput) {
    // Format license key as user types
    licenseKeyInput.addEventListener('input', (e) => {
      let value = e.target.value.toUpperCase().replace(/[^A-Z0-9]/g, '');

      // Insert hyphens
      if (value.length > 4) {
        value = value.slice(0, 4) + '-' + value.slice(4);
      }
      if (value.length > 9) {
        value = value.slice(0, 9) + '-' + value.slice(9);
      }
      if (value.length > 14) {
        value = value.slice(0, 14) + '-' + value.slice(14);
      }
      if (value.length > 19) {
        value = value.slice(0, 19);
      }

      e.target.value = value;

      // Validate format
      if (value.length === 19) {
        if (isValidLicenseKeyFormat(value)) {
          licenseKeyInput.classList.remove('invalid');
          licenseKeyInput.classList.add('valid');
        } else {
          licenseKeyInput.classList.remove('valid');
          licenseKeyInput.classList.add('invalid');
        }
      } else {
        licenseKeyInput.classList.remove('valid', 'invalid');
      }
    });

    activateLicenseBtn.addEventListener('click', async () => {
      const licenseKey = licenseKeyInput.value.trim();

      if (!licenseKey) {
        showNotification('Please enter a license key', 'error');
        return;
      }

      if (!isValidLicenseKeyFormat(licenseKey)) {
        showNotification('Invalid license key format. Use: XXXX-XXXX-XXXX-XXXX', 'error');
        licenseKeyInput.classList.add('invalid');
        return;
      }

      // Show loading state
      activateLicenseBtn.classList.add('btn-loading');
      activateLicenseBtn.disabled = true;

      const result = await activateLicenseKey(licenseKey);

      activateLicenseBtn.classList.remove('btn-loading');
      activateLicenseBtn.disabled = false;

      if (result.success) {
        showNotification('License activated successfully!');
        licenseKeyInput.value = '';
        licenseKeyInput.classList.remove('valid', 'invalid');
        updateLicenseDisplay(result.license);

        // Notify background script
        chrome.runtime.sendMessage({
          action: 'licenseUpdated',
          license: result.license
        });
      } else {
        showNotification('Activation failed: ' + (result.error || 'Unknown error'), 'error');
        licenseKeyInput.classList.add('invalid');
      }
    });

    // Allow Enter key to activate
    licenseKeyInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        activateLicenseBtn.click();
      }
    });
  }

  // Refresh License button
  const refreshLicenseBtn = document.getElementById('btn-refresh-license');
  if (refreshLicenseBtn) {
    refreshLicenseBtn.addEventListener('click', async () => {
      refreshLicenseBtn.classList.add('btn-loading');
      refreshLicenseBtn.disabled = true;

      // Check desktop app and try to sync
      const status = await detectDesktopApp();
      updateDesktopAppStatus(status);

      if (status.connected) {
        const result = await syncLicenseFromDesktop();
        if (result.success) {
          showNotification('License refreshed');
          updateLicenseDisplay(result.license);

          chrome.runtime.sendMessage({
            action: 'licenseUpdated',
            license: result.license
          });
        } else {
          // Just reload from storage
          const license = await loadLicense();
          updateLicenseDisplay(license);
          showNotification('License status refreshed');
        }
      } else {
        // Just reload from storage
        const license = await loadLicense();
        updateLicenseDisplay(license);
        showNotification('License status refreshed');
      }

      refreshLicenseBtn.classList.remove('btn-loading');
      refreshLicenseBtn.disabled = false;
    });
  }

  // Deactivate License button
  const deactivateLicenseBtn = document.getElementById('btn-deactivate-license');
  if (deactivateLicenseBtn) {
    deactivateLicenseBtn.addEventListener('click', async () => {
      if (!confirm('Are you sure you want to deactivate your license? You will lose access to premium features.')) {
        return;
      }

      const result = await deactivateLicense();

      if (result.success) {
        showNotification('License deactivated');
        const license = await loadLicense();
        updateLicenseDisplay(license);
      } else {
        showNotification('Failed to deactivate license', 'error');
      }
    });
  }
}

// ============================================
// INITIALIZATION
// ============================================

/**
 * Initialize the options page
 */
async function init() {
  try {
    // Load all settings
    const settings = await loadSettings();
    const stats = await loadStats();
    const license = await loadLicense();

    // Apply theme
    applyTheme(settings.theme);

    // Update all UI elements
    updateToggles(settings.config);
    updateFilterLists(settings.filterLists);
    updateSponsorBlock(settings.sponsorblock);
    updateStatsDisplay(stats);
    updateWhitelistDisplay(settings.whitelist);
    updateLicenseDisplay(license);

    // Load custom rules into textarea
    const rulesTextarea = document.getElementById('custom-rules');
    if (rulesTextarea && settings.rules) {
      rulesTextarea.value = settings.rules;
    }

    // Setup event handlers
    setupEventHandlers();

    // Check for desktop app connection in the background
    detectDesktopApp().then(status => {
      updateDesktopAppStatus(status);

      // If connected and no premium license, try auto-sync
      if (status.connected && license.tier === 'free') {
        syncLicenseFromDesktop().then(result => {
          if (result.success && result.license.tier !== 'free') {
            updateLicenseDisplay(result.license);
            showNotification('License synced from desktop app!');
          }
        });
      }
    });

    // Listen for storage changes from other contexts
    chrome.storage.onChanged.addListener((changes, areaName) => {
      if (areaName === 'local' && changes[STORAGE_KEYS.STATS]) {
        updateStatsDisplay(changes[STORAGE_KEYS.STATS].newValue);
      }
      if (areaName === 'sync' && changes[STORAGE_KEYS.LICENSE]) {
        updateLicenseDisplay(changes[STORAGE_KEYS.LICENSE].newValue);
      }
    });

    console.log('StreamShield Options page initialized');

  } catch (error) {
    console.error('Failed to initialize options page:', error);
    showNotification('Failed to load settings', 'error');
  }
}

// Run initialization when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
