/**
 * StreamShield Element Picker
 *
 * A visual element selection tool that allows users to click on page elements
 * and create custom CSS-based blocking rules. The picker overlays a highlight
 * on hovered elements, shows selector information, and provides confirm/cancel
 * controls for the blocking action.
 *
 * This script is injected on-demand by the background script when the user
 * clicks "Launch Element Picker" in the options page.
 *
 * PREMIUM FEATURE: Element picker is a premium-only feature.
 */

(function() {
    'use strict';

    // =========================================================================
    // LOGGING UTILITIES
    // =========================================================================

    function log(...args) {
        console.log('[StreamShield Picker]', ...args);
    }

    function warn(...args) {
        console.warn('[StreamShield Picker]', ...args);
    }

    function error(...args) {
        console.error('[StreamShield Picker]', ...args);
    }

    // =========================================================================
    // PREMIUM CHECK
    // =========================================================================

    /**
     * Check if user has premium access for element picker
     * @returns {Promise<boolean>}
     */
    async function checkPremiumAccess() {
        return new Promise((resolve) => {
            if (typeof chrome === 'undefined' || !chrome.storage) {
                log('Chrome storage not available, allowing access');
                resolve(true); // Allow if we can't check
                return;
            }

            try {
                chrome.storage.sync.get(['streamshieldLicense'], (result) => {
                    if (chrome.runtime.lastError) {
                        warn('Failed to check premium status:', chrome.runtime.lastError.message);
                        resolve(false);
                        return;
                    }

                    if (result.streamshieldLicense) {
                        const license = result.streamshieldLicense;
                        // Check if still valid
                        if (license.lifetime) {
                            log('Lifetime premium license active');
                            resolve(true);
                            return;
                        }
                        if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
                            log('Premium license active');
                            resolve(true);
                            return;
                        }
                    }

                    log('No valid premium license found');
                    resolve(false);
                });
            } catch (e) {
                error('Exception checking premium status:', e.message);
                resolve(false);
            }
        });
    }

    /**
     * Show premium required notification
     */
    function showPremiumRequiredNotice() {
        // Create notification overlay
        const overlay = document.createElement('div');
        overlay.id = 'streamshield-premium-notice';
        overlay.style.cssText = [
            'position: fixed',
            'top: 0',
            'left: 0',
            'right: 0',
            'bottom: 0',
            'background: rgba(0, 0, 0, 0.8)',
            'z-index: 2147483647',
            'display: flex',
            'align-items: center',
            'justify-content: center',
            'font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif'
        ].join('; ');

        const dialog = document.createElement('div');
        dialog.style.cssText = [
            'background: linear-gradient(135deg, #1a1a2e, #16213e)',
            'border-radius: 16px',
            'padding: 32px',
            'max-width: 400px',
            'text-align: center',
            'box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5)',
            'border: 1px solid rgba(255, 255, 255, 0.1)'
        ].join('; ');

        const icon = document.createElement('div');
        icon.textContent = '\uD83D\uDD12'; // Lock emoji
        icon.style.cssText = 'font-size: 48px; margin-bottom: 16px;';

        const title = document.createElement('h2');
        title.textContent = 'Premium Feature';
        title.style.cssText = 'color: #fff; margin: 0 0 12px 0; font-size: 24px;';

        const message = document.createElement('p');
        message.textContent = 'Element Picker is a premium feature. Upgrade to unlock custom element blocking and more!';
        message.style.cssText = 'color: #aaa; margin: 0 0 24px 0; font-size: 14px; line-height: 1.5;';

        const upgradeBtn = document.createElement('button');
        upgradeBtn.textContent = 'Upgrade to Premium';
        upgradeBtn.style.cssText = [
            'background: linear-gradient(135deg, #00C853, #00E676)',
            'color: #000',
            'border: none',
            'padding: 12px 24px',
            'border-radius: 8px',
            'font-size: 14px',
            'font-weight: 600',
            'cursor: pointer',
            'margin-right: 12px',
            'transition: transform 0.2s'
        ].join('; ');
        upgradeBtn.onmouseover = function() { upgradeBtn.style.transform = 'scale(1.05)'; };
        upgradeBtn.onmouseout = function() { upgradeBtn.style.transform = 'scale(1)'; };
        upgradeBtn.onclick = function() {
            window.open('https://streamshield.io/premium', '_blank');
            overlay.remove();
        };

        const closeBtn = document.createElement('button');
        closeBtn.textContent = 'Close';
        closeBtn.style.cssText = [
            'background: rgba(255, 255, 255, 0.1)',
            'color: #fff',
            'border: none',
            'padding: 12px 24px',
            'border-radius: 8px',
            'font-size: 14px',
            'cursor: pointer',
            'transition: background 0.2s'
        ].join('; ');
        closeBtn.onmouseover = function() { closeBtn.style.background = 'rgba(255, 255, 255, 0.2)'; };
        closeBtn.onmouseout = function() { closeBtn.style.background = 'rgba(255, 255, 255, 0.1)'; };
        closeBtn.onclick = function() { overlay.remove(); };

        dialog.appendChild(icon);
        dialog.appendChild(title);
        dialog.appendChild(message);
        dialog.appendChild(upgradeBtn);
        dialog.appendChild(closeBtn);
        overlay.appendChild(dialog);
        document.body.appendChild(overlay);

        // Auto-remove after 10 seconds
        setTimeout(function() {
            if (overlay.parentNode) {
                overlay.remove();
            }
        }, 10000);

        log('Premium required notice displayed');
    }

    // =========================================================================
    // CONFIGURATION
    // =========================================================================

    const CONFIG = {
        // Z-index for all picker UI elements (maximum possible value)
        zIndex: 2147483647,

        // Highlight overlay styling
        highlightColor: 'rgba(255, 0, 0, 0.3)',
        highlightBorder: '2px dashed #ff0000',

        // Preview mode styling (after selection)
        previewColor: 'rgba(255, 0, 0, 0.5)',
        previewBorder: '3px solid #ff0000',

        // Tooltip styling
        tooltipBackground: 'rgba(0, 0, 0, 0.9)',
        tooltipColor: '#00ff88',
        tooltipFontFamily: '"JetBrains Mono", "Fira Code", "Consolas", monospace',

        // Animation durations (ms)
        animationDuration: 150,

        // Elements that should never be blocked
        forbiddenTags: ['html', 'body', 'head'],
        forbiddenIds: [
            'streamshield-picker-overlay',
            'streamshield-picker-highlight',
            'streamshield-picker-tooltip',
            'streamshield-picker-controls',
            'streamshield-picker-preview'
        ]
    };

    // =========================================================================
    // STATE
    // =========================================================================

    let state = {
        isActive: false,
        selectedElement: null,
        generatedSelector: null,
        hoveredElement: null
    };

    // =========================================================================
    // UI ELEMENTS
    // =========================================================================

    let ui = {
        overlay: null,
        highlight: null,
        tooltip: null,
        controls: null,
        preview: null,
        style: null
    };

    // =========================================================================
    // UTILITY FUNCTIONS
    // =========================================================================

    /**
     * Generates a unique CSS selector for the given element.
     * Priority: ID > unique class combination > tag with nth-child
     *
     * @param {HTMLElement} element - The target element
     * @returns {string} A CSS selector that uniquely identifies the element
     */
    function generateSelector(element) {
        if (!element || element.nodeType !== Node.ELEMENT_NODE) {
            return null;
        }

        const tag = element.tagName.toLowerCase();

        // Don't allow blocking forbidden elements
        if (CONFIG.forbiddenTags.includes(tag)) {
            return null;
        }

        // Option 1: Use ID if available and unique
        if (element.id && !CONFIG.forbiddenIds.includes(element.id)) {
            const idSelector = '#' + CSS.escape(element.id);
            if (document.querySelectorAll(idSelector).length === 1) {
                return idSelector;
            }
        }

        // Option 2: Build a class-based selector
        if (element.classList.length > 0) {
            const classes = Array.from(element.classList)
                .filter(function(cls) {
                    // Filter out dynamic/random classes
                    return cls.length > 2 &&
                           !cls.match(/^[a-z]{1,2}[0-9]+$/i) && // Skip short random classes
                           !cls.match(/^_[a-zA-Z0-9]+$/); // Skip underscore-prefixed generated classes
                })
                .slice(0, 3); // Use at most 3 classes

            if (classes.length > 0) {
                const classSelector = tag + '.' + classes.map(function(c) { return CSS.escape(c); }).join('.');

                // Check if this selector is unique enough (matches <= 5 elements)
                const matches = document.querySelectorAll(classSelector);
                if (matches.length === 1) {
                    return classSelector;
                }

                // If not unique, try adding parent context
                if (matches.length <= 10) {
                    const parent = element.parentElement;
                    if (parent && parent.tagName.toLowerCase() !== 'body') {
                        const parentSelector = generateParentSelector(parent);
                        if (parentSelector) {
                            const fullSelector = parentSelector + ' > ' + classSelector;
                            if (document.querySelectorAll(fullSelector).length <= 3) {
                                return fullSelector;
                            }
                        }
                    }
                    // Return the class selector even if not perfectly unique
                    return classSelector;
                }
            }
        }

        // Option 3: Use nth-child with parent context
        const parent = element.parentElement;
        if (parent && parent.tagName.toLowerCase() !== 'html') {
            const siblings = Array.from(parent.children);
            const index = siblings.indexOf(element) + 1;
            const nthSelector = tag + ':nth-child(' + index + ')';

            const parentSelector = generateParentSelector(parent);
            if (parentSelector) {
                return parentSelector + ' > ' + nthSelector;
            }

            return nthSelector;
        }

        // Fallback: just the tag name (least specific)
        return tag;
    }

    /**
     * Generates a selector for a parent element (used for context).
     *
     * @param {HTMLElement} element - The parent element
     * @returns {string|null} A selector for the parent
     */
    function generateParentSelector(element) {
        if (!element || CONFIG.forbiddenTags.includes(element.tagName.toLowerCase())) {
            return null;
        }

        const tag = element.tagName.toLowerCase();

        // Use ID if available
        if (element.id && !CONFIG.forbiddenIds.includes(element.id)) {
            return '#' + CSS.escape(element.id);
        }

        // Use significant class
        if (element.classList.length > 0) {
            const mainClass = Array.from(element.classList)
                .find(function(cls) { return cls.length > 3 && !cls.match(/^[a-z]{1,2}[0-9]+$/i); });
            if (mainClass) {
                return tag + '.' + CSS.escape(mainClass);
            }
        }

        // Use just the tag
        return tag;
    }

    /**
     * Gets element info for display in tooltip.
     *
     * @param {HTMLElement} element - The target element
     * @returns {object} Element information
     */
    function getElementInfo(element) {
        if (!element) return null;

        const tag = element.tagName.toLowerCase();
        const id = element.id ? '#' + element.id : '';
        const classes = element.classList.length > 0
            ? '.' + Array.from(element.classList).slice(0, 3).join('.')
            : '';

        // Truncate if too long
        let display = tag + id + classes;
        if (display.length > 50) {
            display = display.substring(0, 47) + '...';
        }

        return {
            tag: tag,
            id: element.id || null,
            classes: Array.from(element.classList),
            display: display,
            rect: element.getBoundingClientRect()
        };
    }

    /**
     * Checks if an element is part of the picker UI.
     *
     * @param {HTMLElement} element - The element to check
     * @returns {boolean} True if element is part of picker UI
     */
    function isPickerElement(element) {
        if (!element) return false;

        return CONFIG.forbiddenIds.some(function(id) {
            const pickerEl = document.getElementById(id);
            return pickerEl && (pickerEl === element || pickerEl.contains(element));
        });
    }

    /**
     * Checks if an element can be blocked.
     *
     * @param {HTMLElement} element - The element to check
     * @returns {boolean} True if element can be blocked
     */
    function canBlockElement(element) {
        if (!element) return false;

        const tag = element.tagName.toLowerCase();

        // Don't allow blocking forbidden elements
        if (CONFIG.forbiddenTags.includes(tag)) {
            return false;
        }

        // Don't allow blocking picker UI
        if (isPickerElement(element)) {
            return false;
        }

        return true;
    }

    // =========================================================================
    // UI CREATION
    // =========================================================================

    /**
     * Creates and injects the picker styles.
     */
    function createStyles() {
        if (ui.style) return;

        ui.style = document.createElement('style');
        ui.style.id = 'streamshield-picker-styles';
        ui.style.textContent = [
            '/* Picker UI Animations */',
            '@keyframes streamshield-fade-in {',
            '    from { opacity: 0; }',
            '    to { opacity: 1; }',
            '}',
            '',
            '@keyframes streamshield-slide-up {',
            '    from {',
            '        opacity: 0;',
            '        transform: translateY(20px);',
            '    }',
            '    to {',
            '        opacity: 1;',
            '        transform: translateY(0);',
            '    }',
            '}',
            '',
            '@keyframes streamshield-pulse {',
            '    0%, 100% { box-shadow: 0 0 0 0 rgba(255, 0, 0, 0.4); }',
            '    50% { box-shadow: 0 0 0 10px rgba(255, 0, 0, 0); }',
            '}',
            '',
            '/* Picker Overlay */',
            '#streamshield-picker-overlay {',
            '    position: fixed;',
            '    top: 0;',
            '    left: 0;',
            '    width: 100%;',
            '    height: 100%;',
            '    z-index: ' + (CONFIG.zIndex - 3) + ';',
            '    pointer-events: none;',
            '    animation: streamshield-fade-in ' + CONFIG.animationDuration + 'ms ease-out;',
            '}',
            '',
            '/* Highlight Box */',
            '#streamshield-picker-highlight {',
            '    position: fixed;',
            '    pointer-events: none;',
            '    z-index: ' + (CONFIG.zIndex - 2) + ';',
            '    background: ' + CONFIG.highlightColor + ';',
            '    border: ' + CONFIG.highlightBorder + ';',
            '    border-radius: 2px;',
            '    transition: all ' + CONFIG.animationDuration + 'ms ease-out;',
            '    box-sizing: border-box;',
            '}',
            '',
            '/* Preview Box (after selection) */',
            '#streamshield-picker-preview {',
            '    position: fixed;',
            '    pointer-events: none;',
            '    z-index: ' + (CONFIG.zIndex - 2) + ';',
            '    background: ' + CONFIG.previewColor + ';',
            '    border: ' + CONFIG.previewBorder + ';',
            '    border-radius: 2px;',
            '    animation: streamshield-pulse 1.5s infinite;',
            '    box-sizing: border-box;',
            '}',
            '',
            '/* Tooltip */',
            '#streamshield-picker-tooltip {',
            '    position: fixed;',
            '    z-index: ' + (CONFIG.zIndex - 1) + ';',
            '    background: ' + CONFIG.tooltipBackground + ';',
            '    color: ' + CONFIG.tooltipColor + ';',
            '    font-family: ' + CONFIG.tooltipFontFamily + ';',
            '    font-size: 12px;',
            '    padding: 8px 12px;',
            '    border-radius: 6px;',
            '    pointer-events: none;',
            '    white-space: nowrap;',
            '    max-width: 400px;',
            '    overflow: hidden;',
            '    text-overflow: ellipsis;',
            '    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);',
            '    border: 1px solid rgba(255, 255, 255, 0.1);',
            '    transition: all ' + CONFIG.animationDuration + 'ms ease-out;',
            '}',
            '',
            '#streamshield-picker-tooltip::before {',
            '    content: "";',
            '    position: absolute;',
            '    top: -6px;',
            '    left: 20px;',
            '    border-left: 6px solid transparent;',
            '    border-right: 6px solid transparent;',
            '    border-bottom: 6px solid ' + CONFIG.tooltipBackground + ';',
            '}',
            '',
            '/* Control Panel */',
            '#streamshield-picker-controls {',
            '    position: fixed;',
            '    bottom: 30px;',
            '    left: 50%;',
            '    transform: translateX(-50%);',
            '    z-index: ' + CONFIG.zIndex + ';',
            '    display: flex;',
            '    gap: 12px;',
            '    padding: 16px 24px;',
            '    background: rgba(10, 10, 20, 0.95);',
            '    border-radius: 12px;',
            '    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5);',
            '    border: 1px solid rgba(255, 255, 255, 0.1);',
            '    animation: streamshield-slide-up ' + CONFIG.animationDuration + 'ms ease-out;',
            '    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;',
            '}',
            '',
            '.streamshield-picker-btn {',
            '    padding: 10px 20px;',
            '    border: none;',
            '    border-radius: 8px;',
            '    font-size: 14px;',
            '    font-weight: 600;',
            '    cursor: pointer;',
            '    transition: all ' + CONFIG.animationDuration + 'ms ease;',
            '    display: flex;',
            '    align-items: center;',
            '    gap: 8px;',
            '}',
            '',
            '.streamshield-picker-btn:hover {',
            '    transform: translateY(-2px);',
            '}',
            '',
            '.streamshield-picker-btn:active {',
            '    transform: translateY(0);',
            '}',
            '',
            '.streamshield-picker-btn-confirm {',
            '    background: linear-gradient(135deg, #00C853, #00E676);',
            '    color: #000;',
            '    box-shadow: 0 4px 15px rgba(0, 200, 83, 0.3);',
            '}',
            '',
            '.streamshield-picker-btn-confirm:hover {',
            '    box-shadow: 0 6px 20px rgba(0, 200, 83, 0.5);',
            '}',
            '',
            '.streamshield-picker-btn-confirm:disabled {',
            '    background: #333;',
            '    color: #666;',
            '    cursor: not-allowed;',
            '    box-shadow: none;',
            '    transform: none;',
            '}',
            '',
            '.streamshield-picker-btn-cancel {',
            '    background: linear-gradient(135deg, #424242, #616161);',
            '    color: #fff;',
            '    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);',
            '}',
            '',
            '.streamshield-picker-btn-cancel:hover {',
            '    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.5);',
            '}',
            '',
            '.streamshield-picker-status {',
            '    color: #888;',
            '    font-size: 13px;',
            '    display: flex;',
            '    align-items: center;',
            '    padding: 0 12px;',
            '    border-left: 1px solid rgba(255, 255, 255, 0.1);',
            '    margin-left: 8px;',
            '}',
            '',
            '.streamshield-picker-selector-display {',
            '    color: #00ff88;',
            '    font-family: ' + CONFIG.tooltipFontFamily + ';',
            '    font-size: 12px;',
            '    max-width: 200px;',
            '    overflow: hidden;',
            '    text-overflow: ellipsis;',
            '    white-space: nowrap;',
            '}'
        ].join('\n');

        document.head.appendChild(ui.style);
    }

    /**
     * Creates the overlay element.
     */
    function createOverlay() {
        if (ui.overlay) return;

        ui.overlay = document.createElement('div');
        ui.overlay.id = 'streamshield-picker-overlay';
        document.body.appendChild(ui.overlay);
    }

    /**
     * Creates the highlight element.
     */
    function createHighlight() {
        if (ui.highlight) return;

        ui.highlight = document.createElement('div');
        ui.highlight.id = 'streamshield-picker-highlight';
        ui.highlight.style.display = 'none';
        document.body.appendChild(ui.highlight);
    }

    /**
     * Creates the preview element (for after selection).
     */
    function createPreview() {
        if (ui.preview) return;

        ui.preview = document.createElement('div');
        ui.preview.id = 'streamshield-picker-preview';
        ui.preview.style.display = 'none';
        document.body.appendChild(ui.preview);
    }

    /**
     * Creates the tooltip element.
     */
    function createTooltip() {
        if (ui.tooltip) return;

        ui.tooltip = document.createElement('div');
        ui.tooltip.id = 'streamshield-picker-tooltip';
        ui.tooltip.style.display = 'none';
        document.body.appendChild(ui.tooltip);
    }

    /**
     * Creates the control panel using safe DOM methods.
     */
    function createControls() {
        if (ui.controls) return;

        // Create main container
        ui.controls = document.createElement('div');
        ui.controls.id = 'streamshield-picker-controls';

        // Create confirm button
        var confirmBtn = document.createElement('button');
        confirmBtn.className = 'streamshield-picker-btn streamshield-picker-btn-confirm';
        confirmBtn.id = 'streamshield-btn-confirm';
        confirmBtn.disabled = true;

        var confirmIcon = document.createElement('span');
        confirmIcon.textContent = '\u2713'; // Checkmark
        confirmBtn.appendChild(confirmIcon);

        var confirmText = document.createTextNode(' Block Element');
        confirmBtn.appendChild(confirmText);

        // Create cancel button
        var cancelBtn = document.createElement('button');
        cancelBtn.className = 'streamshield-picker-btn streamshield-picker-btn-cancel';
        cancelBtn.id = 'streamshield-btn-cancel';

        var cancelIcon = document.createElement('span');
        cancelIcon.textContent = '\u2715'; // X mark
        cancelBtn.appendChild(cancelIcon);

        var cancelText = document.createTextNode(' Cancel (Esc)');
        cancelBtn.appendChild(cancelText);

        // Create status display
        var statusDiv = document.createElement('div');
        statusDiv.className = 'streamshield-picker-status';

        var selectorDisplay = document.createElement('span');
        selectorDisplay.id = 'streamshield-selector-display';
        selectorDisplay.className = 'streamshield-picker-selector-display';
        selectorDisplay.textContent = 'Click an element to select it';
        statusDiv.appendChild(selectorDisplay);

        // Append all elements
        ui.controls.appendChild(confirmBtn);
        ui.controls.appendChild(cancelBtn);
        ui.controls.appendChild(statusDiv);

        document.body.appendChild(ui.controls);

        // Attach event listeners
        confirmBtn.addEventListener('click', confirmBlock);
        cancelBtn.addEventListener('click', deactivate);
    }

    /**
     * Creates all UI elements.
     */
    function createUI() {
        createStyles();
        createOverlay();
        createHighlight();
        createPreview();
        createTooltip();
        createControls();
    }

    /**
     * Removes all UI elements.
     */
    function removeUI() {
        Object.keys(ui).forEach(function(key) {
            if (ui[key] && ui[key].parentNode) {
                ui[key].parentNode.removeChild(ui[key]);
            }
            ui[key] = null;
        });
    }

    // =========================================================================
    // UI UPDATES
    // =========================================================================

    /**
     * Updates the highlight position based on the hovered element.
     *
     * @param {HTMLElement} element - The element being hovered
     */
    function updateHighlight(element) {
        if (!ui.highlight || !element || state.selectedElement) return;

        var rect = element.getBoundingClientRect();

        ui.highlight.style.display = 'block';
        ui.highlight.style.top = (rect.top + window.scrollY) + 'px';
        ui.highlight.style.left = (rect.left + window.scrollX) + 'px';
        ui.highlight.style.width = rect.width + 'px';
        ui.highlight.style.height = rect.height + 'px';
    }

    /**
     * Hides the highlight.
     */
    function hideHighlight() {
        if (ui.highlight) {
            ui.highlight.style.display = 'none';
        }
    }

    /**
     * Updates the preview position based on the selected element.
     *
     * @param {HTMLElement} element - The selected element
     */
    function updatePreview(element) {
        if (!ui.preview || !element) return;

        var rect = element.getBoundingClientRect();

        ui.preview.style.display = 'block';
        ui.preview.style.top = (rect.top + window.scrollY) + 'px';
        ui.preview.style.left = (rect.left + window.scrollX) + 'px';
        ui.preview.style.width = rect.width + 'px';
        ui.preview.style.height = rect.height + 'px';
    }

    /**
     * Updates the tooltip position and content.
     *
     * @param {HTMLElement} element - The element to show info for
     * @param {MouseEvent} event - The mouse event
     */
    function updateTooltip(element, event) {
        if (!ui.tooltip || !element || state.selectedElement) return;

        var info = getElementInfo(element);
        if (!info) {
            hideTooltip();
            return;
        }

        ui.tooltip.textContent = info.display;
        ui.tooltip.style.display = 'block';

        // Position tooltip near cursor
        var tooltipRect = ui.tooltip.getBoundingClientRect();
        var left = event.clientX + 15;
        var top = event.clientY + 20;

        // Keep tooltip on screen
        if (left + tooltipRect.width > window.innerWidth - 20) {
            left = event.clientX - tooltipRect.width - 15;
        }
        if (top + tooltipRect.height > window.innerHeight - 20) {
            top = event.clientY - tooltipRect.height - 20;
        }

        ui.tooltip.style.left = left + 'px';
        ui.tooltip.style.top = top + 'px';
    }

    /**
     * Hides the tooltip.
     */
    function hideTooltip() {
        if (ui.tooltip) {
            ui.tooltip.style.display = 'none';
        }
    }

    /**
     * Updates the selector display in controls.
     *
     * @param {string} selector - The generated selector
     */
    function updateSelectorDisplay(selector) {
        var display = document.getElementById('streamshield-selector-display');
        var confirmBtn = document.getElementById('streamshield-btn-confirm');

        if (display) {
            if (selector) {
                display.textContent = selector;
                display.title = selector;
            } else {
                display.textContent = 'Click an element to select it';
                display.title = '';
            }
        }

        if (confirmBtn) {
            confirmBtn.disabled = !selector;
        }
    }

    // =========================================================================
    // EVENT HANDLERS
    // =========================================================================

    /**
     * Handles mouse movement over the page.
     *
     * @param {MouseEvent} event - The mouse event
     */
    function handleMouseMove(event) {
        if (!state.isActive || state.selectedElement) return;

        var target = document.elementFromPoint(event.clientX, event.clientY);

        if (!target || isPickerElement(target) || !canBlockElement(target)) {
            hideHighlight();
            hideTooltip();
            state.hoveredElement = null;
            return;
        }

        if (target !== state.hoveredElement) {
            state.hoveredElement = target;
            updateHighlight(target);
        }

        updateTooltip(target, event);
    }

    /**
     * Handles click on an element.
     *
     * @param {MouseEvent} event - The click event
     */
    function handleClick(event) {
        if (!state.isActive) return;

        // Always prevent default when picker is active
        event.preventDefault();
        event.stopPropagation();

        var target = document.elementFromPoint(event.clientX, event.clientY);

        // Ignore clicks on picker UI
        if (isPickerElement(target)) {
            return;
        }

        // Ignore clicks on elements that can't be blocked
        if (!canBlockElement(target)) {
            return;
        }

        // Select the element
        selectElement(target);
    }

    /**
     * Handles keydown events.
     *
     * @param {KeyboardEvent} event - The keyboard event
     */
    function handleKeyDown(event) {
        if (!state.isActive) return;

        if (event.key === 'Escape') {
            event.preventDefault();

            // If an element is selected, deselect it
            if (state.selectedElement) {
                deselectElement();
            } else {
                // Otherwise, exit picker mode
                deactivate();
            }
        }
    }

    /**
     * Handles scroll events to update preview position.
     */
    function handleScroll() {
        if (state.selectedElement) {
            updatePreview(state.selectedElement);
        }
    }

    // =========================================================================
    // SELECTION LOGIC
    // =========================================================================

    /**
     * Selects an element for blocking.
     *
     * @param {HTMLElement} element - The element to select
     */
    function selectElement(element) {
        if (!element || !canBlockElement(element)) return;

        state.selectedElement = element;
        state.generatedSelector = generateSelector(element);

        // Update UI
        hideHighlight();
        hideTooltip();
        updatePreview(element);
        updateSelectorDisplay(state.generatedSelector);

        console.log('[StreamShield Picker] Selected element:', state.generatedSelector);
    }

    /**
     * Deselects the currently selected element.
     */
    function deselectElement() {
        state.selectedElement = null;
        state.generatedSelector = null;

        // Update UI
        if (ui.preview) {
            ui.preview.style.display = 'none';
        }
        updateSelectorDisplay(null);
    }

    // =========================================================================
    // RULE MANAGEMENT
    // =========================================================================

    // Backend API configuration
    const API_CONFIG = {
        baseUrl: 'http://localhost:8081',
        timeout: 5000
    };

    /**
     * Confirms blocking of the selected element.
     */
    function confirmBlock() {
        if (!state.selectedElement || !state.generatedSelector) return;

        var domain = window.location.hostname;
        var rule = domain + '##' + state.generatedSelector;

        // Save the rule to storage and backend
        saveRule(rule, domain, state.generatedSelector);

        // Apply the rule immediately
        applyRule(state.generatedSelector);

        // Deactivate the picker
        deactivate();
    }

    /**
     * Syncs a rule to the backend API for cross-device sync.
     *
     * @param {string} domain - The domain for the rule
     * @param {string} selector - The CSS selector
     * @returns {Promise<object|null>} The saved rule from API or null on failure
     */
    async function syncRuleToBackend(domain, selector) {
        try {
            const response = await fetch(`${API_CONFIG.baseUrl}/api/rules/element`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    domain: domain,
                    selector: selector,
                    source: 'element_picker',
                    url: window.location.href
                }),
                signal: AbortSignal.timeout(API_CONFIG.timeout)
            });

            if (response.ok) {
                const result = await response.json();
                log('Rule synced to backend:', result.rule?.id);
                return result.rule;
            } else {
                const error = await response.json();
                warn('Backend sync failed:', error.error || response.status);
                return null;
            }
        } catch (e) {
            // Backend might not be running, silently fail
            warn('Backend sync unavailable:', e.message);
            return null;
        }
    }

    /**
     * Fetches rules for the current domain from the backend API.
     *
     * @param {string} domain - The domain to fetch rules for
     * @returns {Promise<Array>} Array of rules from backend
     */
    async function fetchRulesFromBackend(domain) {
        try {
            const response = await fetch(`${API_CONFIG.baseUrl}/api/rules/elements/${encodeURIComponent(domain)}`, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json'
                },
                signal: AbortSignal.timeout(API_CONFIG.timeout)
            });

            if (response.ok) {
                const data = await response.json();
                log('Fetched', data.rules?.length || 0, 'rules from backend for', domain);
                return data.rules || [];
            }
            return [];
        } catch (e) {
            // Backend might not be running
            warn('Backend fetch unavailable:', e.message);
            return [];
        }
    }

    /**
     * Saves a blocking rule to chrome.storage and syncs to backend.
     *
     * @param {string} rule - The rule in format: domain##selector
     * @param {string} domain - The domain
     * @param {string} selector - The CSS selector
     */
    function saveRule(rule, domain, selector) {
        if (typeof chrome !== 'undefined' && chrome.storage) {
            chrome.storage.local.get(['customElementRules'], function(result) {
                var rules = result.customElementRules || [];

                // Avoid duplicates
                if (rules.indexOf(rule) === -1) {
                    rules.push(rule);

                    chrome.storage.local.set({ customElementRules: rules }, function() {
                        log('Rule saved to local storage:', rule);

                        // Notify background script
                        chrome.runtime.sendMessage({
                            action: 'elementRuleAdded',
                            rule: rule
                        });

                        // Sync to backend API for cross-device sync
                        syncRuleToBackend(domain, selector).then(function(backendRule) {
                            if (backendRule) {
                                // Store the backend rule ID for future sync
                                chrome.storage.local.get(['elementRuleIds'], function(idResult) {
                                    var ruleIds = idResult.elementRuleIds || {};
                                    ruleIds[rule] = backendRule.id;
                                    chrome.storage.local.set({ elementRuleIds: ruleIds });
                                });
                            }
                        });
                    });
                }
            });
        }
    }

    /**
     * Applies a CSS rule to hide the element.
     *
     * @param {string} selector - The CSS selector
     */
    function applyRule(selector) {
        try {
            var elements = document.querySelectorAll(selector);
            elements.forEach(function(el) {
                el.style.setProperty('display', 'none', 'important');
                el.style.setProperty('visibility', 'hidden', 'important');
            });

            // Also add to document styles for persistence
            var style = document.createElement('style');
            style.textContent = selector + ' { display: none !important; visibility: hidden !important; }';
            document.head.appendChild(style);

            console.log('[StreamShield Picker] Rule applied:', selector, '(' + elements.length + ' elements hidden)');
        } catch (error) {
            console.error('[StreamShield Picker] Failed to apply rule:', error);
        }
    }

    // =========================================================================
    // ACTIVATION / DEACTIVATION
    // =========================================================================

    /**
     * Activates the element picker.
     * Checks for premium status before activation.
     */
    async function activate() {
        if (state.isActive) return;

        log('Checking premium status before activation...');

        // Check premium status
        const hasPremium = await checkPremiumAccess();

        if (!hasPremium) {
            log('Premium required for element picker');
            showPremiumRequiredNotice();

            // Notify background script that picker was rejected
            if (typeof chrome !== 'undefined' && chrome.runtime) {
                try {
                    chrome.runtime.sendMessage({
                        action: 'elementPickerRejected',
                        reason: 'premium_required'
                    });
                } catch (e) {
                    // Ignore
                }
            }
            return;
        }

        log('Activating element picker');

        state.isActive = true;
        state.selectedElement = null;
        state.generatedSelector = null;
        state.hoveredElement = null;

        // Create UI
        try {
            createUI();
        } catch (e) {
            error('Failed to create picker UI:', e.message);
            state.isActive = false;
            return;
        }

        // Add event listeners
        document.addEventListener('mousemove', handleMouseMove, true);
        document.addEventListener('click', handleClick, true);
        document.addEventListener('keydown', handleKeyDown, true);
        window.addEventListener('scroll', handleScroll, true);

        // Prevent context menu
        document.addEventListener('contextmenu', handleClick, true);

        log('Element picker activated successfully');
    }

    /**
     * Deactivates the element picker.
     */
    function deactivate() {
        if (!state.isActive) return;

        log('Deactivating element picker');

        state.isActive = false;
        state.selectedElement = null;
        state.generatedSelector = null;
        state.hoveredElement = null;

        // Remove event listeners
        try {
            document.removeEventListener('mousemove', handleMouseMove, true);
            document.removeEventListener('click', handleClick, true);
            document.removeEventListener('keydown', handleKeyDown, true);
            window.removeEventListener('scroll', handleScroll, true);
            document.removeEventListener('contextmenu', handleClick, true);
        } catch (e) {
            warn('Error removing event listeners:', e.message);
        }

        // Remove UI
        try {
            removeUI();
        } catch (e) {
            warn('Error removing UI:', e.message);
        }

        // Notify background script
        if (typeof chrome !== 'undefined' && chrome.runtime) {
            try {
                chrome.runtime.sendMessage({ action: 'elementPickerClosed' });
            } catch (e) {
                // Ignore - extension context may be invalid
            }
        }

        log('Element picker deactivated');
    }

    // =========================================================================
    // MESSAGE HANDLING
    // =========================================================================

    /**
     * Handles messages from the background script.
     */
    function setupMessageHandling() {
        if (typeof chrome === 'undefined' || !chrome.runtime) {
            warn('Chrome runtime not available for message handling');
            return;
        }

        try {
            chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
                try {
                    switch (request.action) {
                        case 'activateElementPicker':
                            // activate() is async, handle properly
                            activate().then(function() {
                                sendResponse({ success: true, active: state.isActive });
                            }).catch(function(e) {
                                error('Activation failed:', e.message);
                                sendResponse({ success: false, error: e.message });
                            });
                            return true; // Keep channel open for async

                        case 'deactivateElementPicker':
                            deactivate();
                            sendResponse({ success: true });
                            break;

                        case 'isElementPickerActive':
                            sendResponse({ active: state.isActive });
                            break;

                        case 'checkElementPickerPremium':
                            checkPremiumAccess().then(function(hasPremium) {
                                sendResponse({ hasPremium: hasPremium });
                            });
                            return true; // Keep channel open for async

                        default:
                            // Unknown action, don't respond
                            break;
                    }
                } catch (e) {
                    error('Message handler error:', e.message);
                    sendResponse({ success: false, error: e.message });
                }
                return true;
            });

            log('Message handling initialized');
        } catch (e) {
            error('Failed to setup message handling:', e.message);
        }
    }

    // Initialize message handling
    setupMessageHandling();

    // =========================================================================
    // APPLY SAVED RULES ON PAGE LOAD
    // =========================================================================

    /**
     * Applies previously saved element rules for this domain.
     * Fetches from both local storage and backend API, merging results.
     */
    async function applySavedRules() {
        if (typeof chrome === 'undefined' || !chrome.storage) return;

        var currentDomain = window.location.hostname;
        var appliedSelectors = new Set(); // Track what we've already applied

        // First apply local rules (faster)
        chrome.storage.local.get(['customElementRules'], function(result) {
            var rules = result.customElementRules || [];

            rules.forEach(function(rule) {
                // Parse rule: domain##selector
                var parts = rule.split('##');
                if (parts.length !== 2) return;

                var domain = parts[0];
                var selector = parts[1];

                // Check if rule applies to this domain
                if (currentDomain === domain || currentDomain.endsWith('.' + domain)) {
                    if (!appliedSelectors.has(selector)) {
                        applyRule(selector);
                        appliedSelectors.add(selector);
                    }
                }
            });
        });

        // Then fetch and apply backend rules (may have rules from other devices)
        try {
            const backendRules = await fetchRulesFromBackend(currentDomain);
            if (backendRules && backendRules.length > 0) {
                // Merge backend rules with local storage
                chrome.storage.local.get(['customElementRules'], function(result) {
                    var localRules = result.customElementRules || [];
                    var newRules = [];

                    backendRules.forEach(function(backendRule) {
                        var selector = backendRule.selector;
                        var domain = backendRule.domain;
                        var rule = domain + '##' + selector;

                        // Apply if not already applied
                        if (!appliedSelectors.has(selector)) {
                            if (currentDomain === domain || currentDomain.endsWith('.' + domain)) {
                                applyRule(selector);
                                appliedSelectors.add(selector);
                            }
                        }

                        // Add to local storage if not already there
                        if (localRules.indexOf(rule) === -1) {
                            newRules.push(rule);
                        }
                    });

                    // Save merged rules back to local storage
                    if (newRules.length > 0) {
                        var mergedRules = localRules.concat(newRules);
                        chrome.storage.local.set({ customElementRules: mergedRules }, function() {
                            log('Merged', newRules.length, 'rules from backend');
                        });
                    }
                });
            }
        } catch (e) {
            // Backend unavailable, continue with local rules only
            warn('Backend sync skipped:', e.message);
        }
    }

    // Apply saved rules when script loads
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            applySavedRules();
        });
    } else {
        applySavedRules();
    }

    // =========================================================================
    // AUTO-ACTIVATION (if triggered directly)
    // =========================================================================

    // If this script was injected with an activation flag, activate immediately
    if (window.__streamshieldActivatePicker) {
        delete window.__streamshieldActivatePicker;
        activate().catch(function(e) {
            error('Auto-activation failed:', e.message);
        });
    }

    log('Element picker script loaded (Premium feature)');

})();
