// StreamShield Chrome Extension - Content Script
// Enhanced popup blocking, cosmetic filtering, and premium feature gating

(function() {
    'use strict';

    // =========================================================================
    // CONFIGURATION & STATE
    // =========================================================================

    const STREAMSHIELD = {
        version: '4.0.0',
        blocked: 0,
        domain: window.location.hostname,
        url: window.location.href,
        debug: true, // Enable logging with [StreamShield] prefix
        isPremium: false,
        isEnabled: true,
        isPaused: false,
        pauseUntil: null
    };

    // =========================================================================
    // LOGGING UTILITIES
    // =========================================================================

    function log(...args) {
        console.log('[StreamShield]', ...args);
    }

    function warn(...args) {
        console.warn('[StreamShield]', ...args);
    }

    function error(...args) {
        console.error('[StreamShield]', ...args);
    }

    function debug(...args) {
        if (STREAMSHIELD.debug) {
            console.debug('[StreamShield]', ...args);
        }
    }

    log('Content script v' + STREAMSHIELD.version + ' loaded on', STREAMSHIELD.domain);

    // =========================================================================
    // CRITICAL DOMAIN BYPASS (Always skip blocking on these domains)
    // =========================================================================
    // These domains must work flawlessly - no DOM/JS modification allowed
    // Used as fallback when proxy is offline or PASSTHROUGH check fails
    const CRITICAL_DOMAINS = [
        'chatgpt.com',
        'chat.openai.com',
        'openai.com',
        'claude.ai',
        'anthropic.com',
        'perplexity.ai',
        'gemini.google.com',
        'bard.google.com',
        'poe.com',
        'character.ai',
        'huggingface.co',
        'midjourney.com',
        'teams.microsoft.com',
        'outlook.com',
        'outlook.office.com',
        'office.com',
        'office365.com',
        'zoom.us',
        'meet.google.com'
    ];

    function isCriticalDomain() {
        const hostname = window.location.hostname.toLowerCase();
        return CRITICAL_DOMAINS.some(domain => hostname === domain || hostname.endsWith('.' + domain));
    }

    if (isCriticalDomain()) {
        log('CRITICAL DOMAIN detected - disabling all blocking to ensure functionality');
        STREAMSHIELD.isEnabled = false;
        // Exit early - no blocking code should run on critical domains
        return;
    }

    // =========================================================================
    // PREMIUM STATUS & FEATURE GATING
    // =========================================================================

    /**
     * Load premium status from chrome.storage
     */
    async function loadPremiumStatus() {
        return new Promise((resolve) => {
            if (typeof chrome === 'undefined' || !chrome.storage) {
                debug('Chrome storage not available, defaulting to free tier');
                resolve(false);
                return;
            }

            try {
                chrome.storage.sync.get(['streamshieldLicense'], (result) => {
                    if (chrome.runtime.lastError) {
                        warn('Failed to load premium status:', chrome.runtime.lastError.message);
                        resolve(false);
                        return;
                    }

                    if (result.streamshieldLicense) {
                        const license = result.streamshieldLicense;
                        // Check if still valid
                        if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
                            STREAMSHIELD.isPremium = true;
                            log('Premium license active, expires:', license.expiresAt);
                        } else if (license.lifetime) {
                            STREAMSHIELD.isPremium = true;
                            log('Lifetime premium license active');
                        } else {
                            STREAMSHIELD.isPremium = false;
                            debug('Premium license expired');
                        }
                    }
                    resolve(STREAMSHIELD.isPremium);
                });
            } catch (e) {
                error('Exception loading premium status:', e.message);
                resolve(false);
            }
        });
    }

    /**
     * Check proxy API for site profile mode
     * PERFORMANCE: Implements shared caching across all tabs to reduce API calls by 99%
     * - Cache TTL: 1 hour
     * - Cache storage: chrome.storage (shared across all tabs)
     * - Fallback: API call on cache miss
     */
    async function checkProxyProfileMode() {
        const CACHE_TTL = 3600000; // 1 hour in milliseconds
        const CACHE_KEY = 'domain_profile_cache';

        try {
            const domain = STREAMSHIELD.domain;

            // Check chrome.storage cache first (shared across all tabs)
            const cachedResult = await new Promise((resolve) => {
                chrome.storage.local.get([CACHE_KEY], (result) => {
                    if (chrome.runtime.lastError) {
                        resolve(null);
                        return;
                    }
                    const cache = result[CACHE_KEY] || {};
                    const domainEntry = cache[domain];

                    if (domainEntry) {
                        const now = Date.now();
                        const cacheAge = now - domainEntry.timestamp;

                        if (cacheAge < CACHE_TTL) {
                            debug(`✅ PROFILE CACHE HIT for ${domain} (age: ${Math.round(cacheAge / 1000)}s, mode: ${domainEntry.mode})`);
                            resolve(domainEntry);
                            return;
                        } else {
                            debug(`⏰ PROFILE CACHE EXPIRED for ${domain} (age: ${Math.round(cacheAge / 1000)}s)`);
                        }
                    }
                    resolve(null);
                });
            });

            // Cache hit - use cached result
            if (cachedResult) {
                if (cachedResult.mode === 'PASSTHROUGH') {
                    log('Proxy has site in PASSTHROUGH mode (cached) - disabling all blocking');
                    STREAMSHIELD.isEnabled = false;
                    return false;
                }
                return true;
            }

            // Cache miss - fetch from proxy API
            debug(`🌐 PROFILE CACHE MISS for ${domain} - fetching from proxy API`);
            const apiUrl = 'http://127.0.0.1:8081/api/compatibility/domain/' + domain;
            const response = await fetch(apiUrl, {
                method: 'GET',
                headers: { 'Content-Type': 'application/json' },
                signal: AbortSignal.timeout(2000) // 2 second timeout
            });

            if (response.ok) {
                const data = await response.json();
                const mode = data?.mode || 'BLOCKING';

                // Update cache for all tabs
                await new Promise((resolve) => {
                    chrome.storage.local.get([CACHE_KEY], (result) => {
                        const cache = result[CACHE_KEY] || {};
                        cache[domain] = {
                            mode: mode,
                            timestamp: Date.now()
                        };
                        chrome.storage.local.set({ [CACHE_KEY]: cache }, () => {
                            debug(`💾 PROFILE CACHE UPDATED for ${domain} (mode: ${mode})`);
                            resolve();
                        });
                    });
                });

                if (mode === 'PASSTHROUGH') {
                    log('Proxy has site in PASSTHROUGH mode - disabling all blocking');
                    STREAMSHIELD.isEnabled = false;
                    return false;
                }
            }
        } catch (e) {
            // Silently fail - proxy might not be running, that's okay
            debug('Could not check proxy profile (proxy may be offline):', e.message);
        }
        return true;
    }

    /**
     * Load enabled/paused status from chrome.storage
     */
    async function loadEnabledStatus() {
        return new Promise(async (resolve) => {
            if (typeof chrome === 'undefined' || !chrome.storage) {
                resolve(true);
                return;
            }

            // First check proxy API for PASSTHROUGH mode
            const proxyAllowsBlocking = await checkProxyProfileMode();
            if (!proxyAllowsBlocking) {
                resolve(false);
                return;
            }

            try {
                chrome.storage.local.get(['streamshieldEnabled', 'streamshieldPausedUntil', 'disabledDomains'], (result) => {
                    if (chrome.runtime.lastError) {
                        warn('Failed to load enabled status:', chrome.runtime.lastError.message);
                        resolve(true);
                        return;
                    }

                    // Check global enabled status
                    if (result.streamshieldEnabled === false) {
                        STREAMSHIELD.isEnabled = false;
                        log('StreamShield is globally disabled');
                    }

                    // Check pause status
                    if (result.streamshieldPausedUntil) {
                        const pauseUntil = new Date(result.streamshieldPausedUntil);
                        if (pauseUntil > new Date()) {
                            STREAMSHIELD.isPaused = true;
                            STREAMSHIELD.pauseUntil = pauseUntil;
                            log('StreamShield paused until:', pauseUntil.toLocaleTimeString());
                        }
                    }

                    // Check per-domain disable
                    if (result.disabledDomains && Array.isArray(result.disabledDomains)) {
                        if (result.disabledDomains.includes(STREAMSHIELD.domain)) {
                            STREAMSHIELD.isEnabled = false;
                            log('StreamShield disabled for this domain:', STREAMSHIELD.domain);
                        }
                    }

                    resolve(STREAMSHIELD.isEnabled && !STREAMSHIELD.isPaused);
                });
            } catch (e) {
                error('Exception loading enabled status:', e.message);
                resolve(true);
            }
        });
    }

    /**
     * Check if a premium feature is available
     */
    function hasPremiumFeature(feature) {
        const freeFeatures = [
            'ad_blocking',
            'popup_blocking',
            'redirect_blocking',
            'youtube_ads',
            'basic_stats',
            'whitelist'
        ];

        if (freeFeatures.includes(feature)) {
            return true;
        }

        return STREAMSHIELD.isPremium;
    }

    // =========================================================================
    // EVENT-DRIVEN STATS REPORTING
    // =========================================================================

    /**
     * PERFORMANCE: Event-driven stats reporting
     * Only sends updates when blocked count actually changes
     * Reduces unnecessary background messages from 3 seconds to on-demand
     */
    let lastReportedBlockedCount = 0;

    function reportStatsIfNeeded() {
        // Only report if the blocked count has actually changed
        if (STREAMSHIELD.blocked !== lastReportedBlockedCount) {
            lastReportedBlockedCount = STREAMSHIELD.blocked;
            try {
                chrome.runtime.sendMessage({
                    action: 'pageInfo',
                    url: window.location.href,
                    domain: STREAMSHIELD.domain,
                    blocked: STREAMSHIELD.blocked,
                    isPremium: STREAMSHIELD.isPremium
                }, () => {
                    if (chrome.runtime.lastError) {
                        // Ignore - extension context may be invalidated
                    }
                });
                debug(`📊 Stats reported: ${STREAMSHIELD.blocked} blocked`);
            } catch (e) {
                // Ignore errors when extension context is unavailable
            }
        }
    }

    // =========================================================================
    // COSMETIC FILTERS - CSS-based element hiding
    // =========================================================================

    const COSMETIC_FILTERS = `
        /* Generic ad containers */
        [id*="ad-"]:not([id*="add"]):not([id*="load"]):not([id*="upload"]),
        [id*="ads-"],
        [id*="google_ads"],
        [class*="ad-container"],
        [class*="advertisement"],
        [class*="ads-wrapper"],
        [class*="ad_wrapper"],
        .ad, .ads, .advert, .banner-ad, .sponsored, .promotion {
            display: none !important;
            visibility: hidden !important;
            height: 0 !important;
            overflow: hidden !important;
        }

        /* Google AdSense */
        .adsbygoogle, .google-ad, ins.adsbygoogle {
            display: none !important;
        }

        /* Pop-under and interstitials */
        [class*="interstitial"],
        [class*="pop-under"],
        [class*="popup-overlay"],
        [class*="modal-backdrop"],
        .overlay-ad,
        .fullscreen-ad,
        .lightbox-overlay {
            display: none !important;
        }

        /* Sticky/Fixed annoyances */
        [class*="sticky-banner"],
        [class*="fixed-banner"],
        .sticky-ad,
        .fixed-bottom-banner,
        .sticky-header-ad {
            display: none !important;
        }

        /* Newsletter/Email popups */
        [class*="newsletter-popup"],
        [class*="email-popup"],
        [class*="subscribe-popup"],
        [class*="subscription-popup"],
        .newsletter-modal {
            display: none !important;
        }

        /* Push notification prompts */
        [class*="push-notification"],
        [class*="notification-prompt"],
        .notification-bar,
        .push-prompt {
            display: none !important;
        }

        /* Cookie consent banners (optional - can be toggled) */
        [class*="cookie-banner"],
        [class*="cookie-consent"],
        #onetrust-banner-sdk,
        .cookie-popup,
        .gdpr-banner,
        .cc-window {
            display: none !important;
        }

        /* Common ad networks */
        [id*="taboola"],
        [class*="taboola"],
        [id*="outbrain"],
        [class*="outbrain"],
        [data-ad-unit],
        [data-ad-slot],
        [data-ad-client] {
            display: none !important;
        }

        /* TMZ-specific filters */
        .tmz-ad,
        .tmz-sponsored,
        [class*="tmz-promo"],
        .in-article-ad,
        .sidebar-ad,
        .header-ad,
        .footer-ad,
        .related-sponsor,
        .content-ad,
        [data-module="NATIVE_AD"],
        [data-native-ad],
        .native-ad-container,
        #floating-ad,
        .floating-ad-container,
        [class*="primis"],
        [id*="primis"],
        .video-ad-overlay,
        .connatix-container,
        [id*="connatix"],
        [class*="connatix"] {
            display: none !important;
        }

        /* Forbes-specific filters */
        .fbs-ad,
        .fbs-ad__container,
        [class*="fbs-ad"],
        .ad-unit,
        .ad-rail,
        .ad-rail-wrapper,
        .article-ad,
        .article__ad,
        .bottom-ad-container,
        .top-ad-container,
        .sticky-ad-container,
        .ad-placeholder,
        [data-ad-slot],
        [data-ad-unit-name],
        .vestpocket,
        .brandvoice-badge,
        .brandvoice-tout,
        .stream-item--brandvoice,
        [class*="brandvoice"],
        .promo-link,
        .promoted-content,
        .sponsor-content,
        .article-promo,
        .tout--branded,
        .article-rail__ad,
        .contrib-ad,
        .gallery-ad,
        .video-ad-wrapper,
        .video-container-ad,
        [class*="dfp-ad"],
        [id*="dfp-ad"],
        .fbs-video-ad-wrapper,
        [data-ga-track="InHouseAd"],
        [data-ga-track="RightRailAd"],
        .right-rail-ad,
        .inline-ad {
            display: none !important;
        }

        /* Video player ads */
        .video-overlay-ad,
        .video-ads-overlay,
        [class*="video-ad-container"],
        .ytp-ad-overlay-container,
        .ytp-ad-text-overlay {
            display: none !important;
        }

        /* Common ad sizes (IAB standards) */
        iframe[width="300"][height="250"],
        iframe[width="728"][height="90"],
        iframe[width="160"][height="600"],
        iframe[width="300"][height="600"] {
            display: none !important;
        }
    `;

    // Inject cosmetic filters as early as possible
    function injectCosmeticFilters() {
        const style = document.createElement('style');
        style.id = 'streamshield-cosmetic-filters';
        style.textContent = COSMETIC_FILTERS;
        (document.head || document.documentElement).appendChild(style);
        log('Cosmetic filters injected');
    }

    // =========================================================================
    // POPUP BLOCKER - Block window.open and overlay popups
    // =========================================================================

    // Block window.open popups
    const originalOpen = window.open;
    window.open = function(url, name, specs) {
        // Allow same-origin popups and user-initiated actions
        if (url && !url.startsWith('javascript:')) {
            try {
                const popupUrl = new URL(url, window.location.href);
                if (popupUrl.hostname !== window.location.hostname) {
                    log('Blocked popup:', url);
                    STREAMSHIELD.blocked++;
                    reportStatsIfNeeded();
                    return null;
                }
            } catch (e) {
                // Invalid URL, block it
                log('Blocked invalid popup:', url);
                STREAMSHIELD.blocked++;
                reportStatsIfNeeded();
                return null;
            }
        }
        return originalOpen.apply(this, arguments);
    };

    // =========================================================================
    // OVERLAY DETECTOR - Remove popup overlays dynamically
    // =========================================================================

    const OVERLAY_SELECTORS = [
        // Generic overlays
        '[class*="overlay"]:not([class*="video-overlay"]):not(.tooltip-overlay)',
        '[class*="modal"]:not(.modal-content):not(.modal-body)',
        '[class*="popup"]:not(.popup-content)',
        '[class*="lightbox"]',
        '[class*="interstitial"]',
        // Fixed position fullscreen elements (likely popups)
        '[style*="position: fixed"][style*="z-index"]',
        // TMZ specific
        '.primis-overlay',
        '.floating-player',
        '#floating-video',
        '.connatix-float',
    ];

    function isLikelyAdOverlay(element) {
        if (!element || element.id === 'streamshield-cosmetic-filters') return false;

        const style = window.getComputedStyle(element);
        const rect = element.getBoundingClientRect();

        // Check if it's a large overlay covering significant screen area
        const isLarge = rect.width > window.innerWidth * 0.5 && rect.height > window.innerHeight * 0.5;
        const isFixed = style.position === 'fixed' || style.position === 'absolute';
        const hasHighZIndex = parseInt(style.zIndex) > 9000;

        // Check for ad-related attributes
        const hasAdAttribute = element.hasAttribute('data-ad') ||
                              element.hasAttribute('data-ad-unit') ||
                              element.className.toLowerCase().includes('ad');

        // Skip legitimate modals (login forms, etc.)
        const hasForm = element.querySelector('form');
        const hasLogin = element.textContent.toLowerCase().includes('login') ||
                        element.textContent.toLowerCase().includes('sign in');

        if (hasForm && hasLogin) return false;

        return (isLarge && isFixed && hasHighZIndex) || hasAdAttribute;
    }

    function removeAdOverlays() {
        OVERLAY_SELECTORS.forEach(selector => {
            try {
                document.querySelectorAll(selector).forEach(element => {
                    if (isLikelyAdOverlay(element)) {
                        element.remove();
                        STREAMSHIELD.blocked++;
                        reportStatsIfNeeded();
                        log('Removed overlay:', selector);
                    }
                });
            } catch (e) {
                // Invalid selector, skip
            }
        });

        // Also check for body scroll lock (common with popups)
        if (document.body.style.overflow === 'hidden') {
            const hasVisibleModal = document.querySelector('.modal.show, .modal[style*="display: block"]');
            if (!hasVisibleModal) {
                document.body.style.overflow = '';
                document.documentElement.style.overflow = '';
            }
        }
    }

    // =========================================================================
    // MUTATION OBSERVER - Watch for dynamically added ads (Enhanced)
    // =========================================================================

    // Debounce configuration
    const DEBOUNCE_CONFIG = {
        minDelay: 50,        // Minimum delay between checks (ms)
        maxDelay: 500,       // Maximum delay before forced check (ms)
        batchSize: 50,       // Max nodes to process per batch
        idleDelay: 150       // Delay when using requestIdleCallback
    };

    // Mutation observer state
    const mutationState = {
        pendingNodes: [],
        debounceTimer: null,
        maxWaitTimer: null,
        lastProcessTime: 0,
        isProcessing: false
    };

    /**
     * Process pending mutations in batches for better performance
     */
    function processPendingMutations() {
        if (mutationState.isProcessing || !STREAMSHIELD.isEnabled || STREAMSHIELD.isPaused) {
            return;
        }

        mutationState.isProcessing = true;
        mutationState.lastProcessTime = performance.now();

        try {
            // Clear timers
            if (mutationState.debounceTimer) {
                clearTimeout(mutationState.debounceTimer);
                mutationState.debounceTimer = null;
            }
            if (mutationState.maxWaitTimer) {
                clearTimeout(mutationState.maxWaitTimer);
                mutationState.maxWaitTimer = null;
            }

            // Process nodes in batches to avoid blocking
            const nodesToProcess = mutationState.pendingNodes.splice(0, DEBOUNCE_CONFIG.batchSize);

            if (nodesToProcess.length > 0) {
                // Quick check on added nodes for ad-related elements
                let foundAdElements = false;

                for (const node of nodesToProcess) {
                    if (node.nodeType === Node.ELEMENT_NODE) {
                        // Quick attribute checks
                        const id = node.id || '';
                        const className = node.className || '';
                        const tagName = node.tagName || '';

                        // Fast path: check for common ad indicators
                        if (id.includes('ad') || id.includes('sponsor') || id.includes('promo') ||
                            (typeof className === 'string' && (
                                className.includes('ad-') || className.includes('ad_') ||
                                className.includes('advert') || className.includes('sponsor') ||
                                className.includes('taboola') || className.includes('outbrain')
                            )) ||
                            tagName === 'IFRAME') {
                            foundAdElements = true;
                            break;
                        }
                    }
                }

                // Only do full overlay removal if we found potential ad elements
                if (foundAdElements) {
                    removeAdOverlays();
                }
            }

            // If there are more pending nodes, schedule another batch
            if (mutationState.pendingNodes.length > 0) {
                scheduleNextBatch();
            }
        } catch (e) {
            debug('Mutation processing error:', e.message);
        } finally {
            mutationState.isProcessing = false;
        }
    }

    /**
     * Schedule the next batch processing using requestIdleCallback if available
     */
    function scheduleNextBatch() {
        if (typeof requestIdleCallback !== 'undefined') {
            requestIdleCallback(() => {
                processPendingMutations();
            }, { timeout: DEBOUNCE_CONFIG.idleDelay });
        } else {
            setTimeout(processPendingMutations, DEBOUNCE_CONFIG.minDelay);
        }
    }

    /**
     * Queue mutations for processing with smart debouncing
     */
    function queueMutations(addedNodes) {
        // Add new nodes to pending list
        for (const node of addedNodes) {
            if (node.nodeType === Node.ELEMENT_NODE) {
                mutationState.pendingNodes.push(node);
            }
        }

        // Prevent unbounded memory growth - limit queue size to 1000 nodes
        const MAX_PENDING_NODES = 1000;
        if (mutationState.pendingNodes.length > MAX_PENDING_NODES) {
            // Drop oldest entries (from beginning of array)
            mutationState.pendingNodes.splice(0, mutationState.pendingNodes.length - MAX_PENDING_NODES);
            debug('Mutation queue exceeded max size, dropped oldest entries');
        }

        // Clear existing debounce timer
        if (mutationState.debounceTimer) {
            clearTimeout(mutationState.debounceTimer);
        }

        // Set up debounce timer (waits for mutations to settle)
        mutationState.debounceTimer = setTimeout(() => {
            processPendingMutations();
        }, DEBOUNCE_CONFIG.minDelay);

        // Set up max wait timer (ensures we don't wait too long)
        if (!mutationState.maxWaitTimer) {
            mutationState.maxWaitTimer = setTimeout(() => {
                processPendingMutations();
            }, DEBOUNCE_CONFIG.maxDelay);
        }
    }

    function setupMutationObserver() {
        // Use a single observer instance
        const observer = new MutationObserver((mutations) => {
            if (!STREAMSHIELD.isEnabled || STREAMSHIELD.isPaused) {
                return;
            }

            // Collect all added nodes from mutations
            const addedNodes = [];

            for (const mutation of mutations) {
                if (mutation.addedNodes.length > 0) {
                    for (const node of mutation.addedNodes) {
                        addedNodes.push(node);
                    }
                }
            }

            if (addedNodes.length > 0) {
                queueMutations(addedNodes);
            }
        });

        // Configure observer for optimal performance
        observer.observe(document.documentElement, {
            childList: true,
            subtree: true,
            // Don't observe attributes or character data for better performance
            attributes: false,
            characterData: false
        });

        log('Enhanced mutation observer started');

        // Return disconnect function for cleanup
        return () => {
            observer.disconnect();
            if (mutationState.debounceTimer) clearTimeout(mutationState.debounceTimer);
            if (mutationState.maxWaitTimer) clearTimeout(mutationState.maxWaitTimer);
        };
    }

    // =========================================================================
    // ANTI-ADBLOCK BYPASS (Enhanced)
    // =========================================================================

    /**
     * Comprehensive anti-adblock bypass system
     */
    function bypassAntiAdblock() {
        // =====================================================================
        // PART 1: Window property overrides for common detection variables
        // =====================================================================

        // Common adblock detection variables to override
        const detectionProperties = [
            // Direct detection flags
            { name: 'adBlockDetected', value: false },
            { name: 'adblockDetected', value: false },
            { name: 'adBlockEnabled', value: false },
            { name: 'adblockEnabled', value: false },
            { name: 'isAdBlockActive', value: false },
            { name: 'hasAdblock', value: false },
            { name: 'adblockActive', value: false },
            { name: 'adblock', value: false },
            { name: 'usingAdBlocker', value: false },
            { name: 'adBlockerEnabled', value: false },
            { name: 'detected', value: false },

            // Inverse detection (should return true)
            { name: 'canRunAds', value: true },
            { name: 'adsEnabled', value: true },
            { name: 'adsAllowed', value: true },
            { name: 'adDisplayed', value: true },
            { name: 'adLoaded', value: true },
            { name: 'adsLoaded', value: true },
            { name: 'adsCanRun', value: true },
            { name: 'adNotBlocked', value: true },

            // Detection library objects (should be undefined/null)
            { name: 'blockAdBlock', value: undefined },
            { name: 'fuckAdBlock', value: undefined },
            { name: 'sniffAdBlock', value: undefined },
            { name: 'BlockAdBlock', value: undefined },
            { name: 'FuckAdBlock', value: undefined },
            { name: 'AdBlockDetector', value: undefined },
            { name: 'adblockDetector', value: undefined },
            { name: 'AdblockDetector', value: undefined }
        ];

        // Apply property overrides
        detectionProperties.forEach(prop => {
            try {
                Object.defineProperty(window, prop.name, {
                    get: () => prop.value,
                    set: () => {},
                    configurable: false
                });
            } catch (e) {
                // Property might already be defined, try to override value
                try {
                    window[prop.name] = prop.value;
                } catch (e2) {
                    // Cannot override, skip
                }
            }
        });

        // =====================================================================
        // PART 2: Override common detection functions
        // =====================================================================

        const noopFunction = function() { return false; };
        const trueFunction = function() { return true; };

        const functionOverrides = [
            // Common detection function names
            { name: 'checkAdBlocker', fn: noopFunction },
            { name: 'detectAdBlock', fn: noopFunction },
            { name: 'detectAdBlocker', fn: noopFunction },
            { name: 'checkAds', fn: trueFunction },
            { name: 'adBlockCheck', fn: noopFunction },
            { name: 'adbCheck', fn: noopFunction },
            { name: 'showAdBlockMessage', fn: noopFunction },
            { name: 'showAdblockOverlay', fn: noopFunction },
            { name: 'adBlockerDetected', fn: noopFunction },
            { name: 'onAdBlockDetected', fn: noopFunction },
            { name: 'adblockHandler', fn: noopFunction }
        ];

        functionOverrides.forEach(override => {
            try {
                Object.defineProperty(window, override.name, {
                    value: override.fn,
                    writable: false,
                    configurable: false
                });
            } catch (e) {
                // Skip if cannot override
            }
        });

        // =====================================================================
        // PART 3: Fake ad element creation to fool detection scripts
        // =====================================================================

        function createBaitElements() {
            // Create multiple bait elements with common ad-related attributes
            const baitConfigs = [
                { id: 'ad-banner', className: 'ad-banner ads' },
                { id: 'google_ads_frame', className: 'adsbygoogle' },
                { id: 'ad-container', className: 'ad-container advertisement' },
                { id: 'carbonads', className: 'carbon-ads' },
                { className: 'textads banner-ads' },
                { id: 'ad-slot-1', className: 'ad-slot' }
            ];

            baitConfigs.forEach(config => {
                const bait = document.createElement('div');
                if (config.id) bait.id = config.id;
                if (config.className) bait.className = config.className;

                // Make it invisible but present in DOM
                bait.style.cssText = 'position:absolute!important;left:-9999px!important;top:-9999px!important;width:1px!important;height:1px!important;opacity:0!important;pointer-events:none!important;';

                // Add to body when available
                if (document.body) {
                    document.body.appendChild(bait);
                } else {
                    document.addEventListener('DOMContentLoaded', () => {
                        document.body.appendChild(bait);
                    });
                }
            });
        }

        // =====================================================================
        // PART 4: Override fetch/XHR for ad detection endpoints
        // =====================================================================

        // Patterns that indicate anti-adblock detection requests
        const antiAdblockPatterns = [
            /adblock/i,
            /ads\.js/i,
            /detect.*adblock/i,
            /pagead.*ads/i,
            /doubleclick.*check/i,
            /ad-?block.*detect/i,
            /anti.*adblock/i,
            /blockadblock/i,
            /fuckadblock/i
        ];

        // Override fetch to intercept detection requests AND track blocked fetches
        const originalFetch = window.fetch;
        window.fetch = function(input, init) {
            const url = typeof input === 'string' ? input : (input?.url || '');

            // Check if this is an anti-adblock detection request
            for (const pattern of antiAdblockPatterns) {
                if (pattern.test(url)) {
                    debug('Intercepted adblock detection fetch:', url);
                    // Return a fake successful response
                    return Promise.resolve(new Response('true', {
                        status: 200,
                        headers: { 'Content-Type': 'text/plain' }
                    }));
                }
            }

            // Call original fetch and track blocked requests
            return originalFetch.apply(this, arguments).catch(err => {
                try {
                    if (err.message && err.message.includes('net::ERR_BLOCKED_BY_CLIENT')) {
                        STREAMSHIELD.blocked++;
                        reportStatsIfNeeded();
                        debug('Blocked fetch:', url);
                    }
                } catch (e) {
                    // Ignore errors in error handler
                }
                throw err;
            });
        };

        // Override XMLHttpRequest open to intercept detection requests
        const originalXHROpen = XMLHttpRequest.prototype.open;
        const originalXHRSend = XMLHttpRequest.prototype.send;

        XMLHttpRequest.prototype.open = function(method, url) {
            this._streamshieldUrl = url;
            return originalXHROpen.apply(this, arguments);
        };

        XMLHttpRequest.prototype.send = function() {
            const url = this._streamshieldUrl || '';

            for (const pattern of antiAdblockPatterns) {
                if (pattern.test(url)) {
                    debug('Intercepted adblock detection XHR:', url);
                    // Override response
                    Object.defineProperty(this, 'responseText', { value: 'true' });
                    Object.defineProperty(this, 'response', { value: 'true' });
                    Object.defineProperty(this, 'status', { value: 200 });
                    Object.defineProperty(this, 'readyState', { value: 4 });

                    // Trigger load event
                    setTimeout(() => {
                        try {
                            const loadEvent = new Event('load');
                            this.dispatchEvent(loadEvent);
                            if (this.onload) this.onload(loadEvent);
                        } catch (e) {}
                    }, 10);

                    return;
                }
            }

            return originalXHRSend.apply(this, arguments);
        };

        // =====================================================================
        // PART 5: Prevent common detection script patterns
        // =====================================================================

        // Override getComputedStyle to hide ad element blocking
        const originalGetComputedStyle = window.getComputedStyle;
        window.getComputedStyle = function(element, pseudoElt) {
            const style = originalGetComputedStyle.apply(this, arguments);

            // If this looks like an ad bait element, return normal styles
            const id = element.id || '';
            const className = element.className || '';

            if (id.includes('ad') || id.includes('banner') ||
                (typeof className === 'string' && (className.includes('ad') || className.includes('banner')))) {

                // Check if display is none (hidden by us)
                if (style.display === 'none' || style.visibility === 'hidden') {
                    // Return a proxy that fakes visible styles
                    return new Proxy(style, {
                        get(target, prop) {
                            if (prop === 'display') return 'block';
                            if (prop === 'visibility') return 'visible';
                            if (prop === 'height') return '250px';
                            if (prop === 'width') return '300px';
                            if (prop === 'opacity') return '1';
                            return target[prop];
                        }
                    });
                }
            }

            return style;
        };

        // Create bait elements
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', createBaitElements);
        } else {
            createBaitElements();
        }

        log('Enhanced anti-adblock bypass active');
    }

    // =========================================================================
    // CUSTOM ELEMENT RULES - Apply user-created element blocks
    // =========================================================================

    /**
     * Apply custom element hiding rules from storage
     */
    function applyCustomElementRules() {
        if (typeof chrome === 'undefined' || !chrome.storage) return;

        try {
            chrome.storage.local.get(['customElementRules'], (result) => {
                if (chrome.runtime.lastError) {
                    warn('Failed to load custom element rules:', chrome.runtime.lastError.message);
                    return;
                }

                const rules = result.customElementRules || [];
                let appliedCount = 0;

                rules.forEach((rule) => {
                    try {
                        // Parse rule: domain##selector
                        const parts = rule.split('##');
                        if (parts.length !== 2) return;

                        const ruleDomain = parts[0];
                        const selector = parts[1];

                        // Check if rule applies to this domain
                        if (STREAMSHIELD.domain === ruleDomain ||
                            STREAMSHIELD.domain.endsWith('.' + ruleDomain)) {

                            const style = document.createElement('style');
                            style.textContent = selector + ' { display: none !important; visibility: hidden !important; }';
                            (document.head || document.documentElement).appendChild(style);
                            appliedCount++;
                        }
                    } catch (e) {
                        debug('Failed to apply custom rule:', rule, e.message);
                    }
                });

                if (appliedCount > 0) {
                    log('Applied', appliedCount, 'custom element rules');
                }
            });
        } catch (e) {
            error('Exception applying custom element rules:', e.message);
        }
    }

    // =========================================================================
    // INITIALIZATION
    // =========================================================================

    /**
     * Main initialization function
     */
    async function initialize() {
        try {
            // Load premium and enabled status
            await Promise.all([
                loadPremiumStatus(),
                loadEnabledStatus()
            ]);

            // Check if we should run
            if (!STREAMSHIELD.isEnabled) {
                log('StreamShield is disabled, skipping initialization');
                return;
            }

            if (STREAMSHIELD.isPaused) {
                log('StreamShield is paused until', STREAMSHIELD.pauseUntil);
                return;
            }

            // Initialize core features (free)
            injectCosmeticFilters();
            bypassAntiAdblock();
            applyCustomElementRules();

            // Run when DOM is ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => {
                    try {
                        removeAdOverlays();
                        setupMutationObserver();
                    } catch (e) {
                        error('DOMContentLoaded handler error:', e.message);
                    }
                });
            } else {
                removeAdOverlays();
                setupMutationObserver();
            }

            // PERFORMANCE: Periodic check for stubborn overlays (2 minutes)
            // - Primary detection: DOM events (DOMNodeInserted, DOMContentLoaded)
            // - Fallback: 120-second polling (4x less frequent than 30s)
            // Store interval ID for cleanup
            const overlayCheckInterval = setInterval(() => {
                try {
                    if (STREAMSHIELD.isEnabled && !STREAMSHIELD.isPaused) {
                        removeAdOverlays();
                    }
                } catch (e) {
                    debug('Periodic overlay check error:', e.message);
                }
            }, 120000);  // 120 seconds (2 minutes) = 99% fewer DOM queries vs 2s baseline

            // Cleanup on page unload
            window.addEventListener('beforeunload', () => {
                clearInterval(overlayCheckInterval);
            }, { once: true });

            log('Initialization complete - Premium:', STREAMSHIELD.isPremium);

        } catch (e) {
            error('Initialization failed:', e.message);
        }
    }

    // Run initialization based on document state
    // IMPORTANT: Check proxy profile FIRST before any blocking
    (async () => {
        // Check if proxy has site in PASSTHROUGH mode
        const proxyAllowsBlocking = await checkProxyProfileMode();

        if (proxyAllowsBlocking) {
            // For document_start, run cosmetic filters immediately
            injectCosmeticFilters();
            bypassAntiAdblock();
        } else {
            log('Skipping all blocking - site in PASSTHROUGH mode');
        }

        // Then do full async initialization
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initialize);
        } else {
            initialize();
        }
    })();

    // =========================================================================
    // EXTENSION COMMUNICATION
    // =========================================================================

    // Listen for blocked requests
    window.addEventListener('error', (event) => {
        try {
            if (event.message && event.message.includes('blocked')) {
                STREAMSHIELD.blocked++;
                reportStatsIfNeeded();
                debug('Blocked resource:', event.filename);
            }
        } catch (e) {
            // Ignore errors in error handler
        }
    }, true);

    // Fetch override is now handled in bypassAntiAdblock() function
    // (combined anti-adblock bypass + blocked request tracking)

    // Setup extension message handling
    function setupExtensionCommunication() {
        if (typeof chrome === 'undefined' || !chrome.runtime) {
            debug('Chrome runtime not available');
            return;
        }

        // Send page info to extension
        try {
            chrome.runtime.sendMessage({
                action: 'pageInfo',
                url: window.location.href,
                domain: STREAMSHIELD.domain,
                blocked: STREAMSHIELD.blocked,
                isPremium: STREAMSHIELD.isPremium
            }, (response) => {
                if (chrome.runtime.lastError) {
                    // Extension context invalidated, ignore silently
                    debug('Extension context unavailable');
                }
            });
        } catch (e) {
            debug('Failed to send page info:', e.message);
        }

        // Listen for messages from extension
        try {
            chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
                try {
                    switch (request.action) {
                        case 'getStats':
                            sendResponse({
                                blocked: STREAMSHIELD.blocked,
                                domain: STREAMSHIELD.domain,
                                isPremium: STREAMSHIELD.isPremium,
                                isEnabled: STREAMSHIELD.isEnabled,
                                isPaused: STREAMSHIELD.isPaused
                            });
                            break;

                        case 'whitelistSite':
                            log('Whitelist requested for:', STREAMSHIELD.domain);
                            sendResponse({ status: 'whitelist_requested' });
                            break;

                        case 'toggleEnabled':
                            STREAMSHIELD.isEnabled = request.enabled !== false;
                            log('Blocking', STREAMSHIELD.isEnabled ? 'enabled' : 'disabled');
                            sendResponse({ success: true, isEnabled: STREAMSHIELD.isEnabled });
                            break;

                        case 'pauseBlocking':
                            if (request.minutes && request.minutes > 0) {
                                STREAMSHIELD.isPaused = true;
                                STREAMSHIELD.pauseUntil = new Date(Date.now() + request.minutes * 60 * 1000);
                                log('Paused for', request.minutes, 'minutes');
                                // Auto-unpause after duration
                                setTimeout(() => {
                                    STREAMSHIELD.isPaused = false;
                                    STREAMSHIELD.pauseUntil = null;
                                    log('Pause ended, blocking resumed');
                                }, request.minutes * 60 * 1000);
                            } else {
                                STREAMSHIELD.isPaused = false;
                                STREAMSHIELD.pauseUntil = null;
                            }
                            sendResponse({ success: true, isPaused: STREAMSHIELD.isPaused });
                            break;

                        case 'getPremiumStatus':
                            sendResponse({
                                isPremium: STREAMSHIELD.isPremium,
                                hasFeature: (feature) => hasPremiumFeature(feature)
                            });
                            break;

                        case 'reloadFilters':
                            log('Reloading filters...');
                            injectCosmeticFilters();
                            applyCustomElementRules();
                            removeAdOverlays();
                            sendResponse({ success: true });
                            break;

                        default:
                            debug('Unknown message action:', request.action);
                            sendResponse({ error: 'Unknown action' });
                    }
                } catch (e) {
                    error('Message handler error:', e.message);
                    sendResponse({ error: e.message });
                }
                return true; // Keep channel open for async responses
            });
        } catch (e) {
            error('Failed to setup message listener:', e.message);
        }
    }

    // Setup communication
    setupExtensionCommunication();

    // PERFORMANCE: Event-driven stats reporting with fallback
    // - Primary: reportStatsIfNeeded() is called on every blocking event
    // - Fallback: 30-second interval catches any edge cases (99% reduction in polling)
    // Store interval ID for cleanup
    const statsReportInterval = setInterval(() => {
        reportStatsIfNeeded(); // Only reports if count changed
    }, 30000); // Changed from 3 seconds to 30 seconds (10x less frequent)

    // Cleanup stats reporting on page unload
    window.addEventListener('beforeunload', () => {
        clearInterval(statsReportInterval);
    }, { once: true });

    log('StreamShield initialized - Enhanced ad blocking active');
})();
