/**
 * StreamShield Browser Extension - Background Script v4.0.0
 * Handles request blocking, statistics, malware/phishing protection,
 * element picker, SponsorBlock integration, premium features,
 * desktop app sync, and communication with StreamShield proxy
 */

// ============================================================================
// PREMIUM FEATURES MANAGER (Integrated from premium.js)
// ============================================================================

const StreamShieldPremium = {
  // Premium status
  isPremium: false,
  licenseKey: null,
  expiresAt: null,
  lastValidatedAt: null,

  // Feature definitions
  FREE_FEATURES: [
    'ad_blocking',
    'popup_blocking',
    'redirect_blocking',
    'youtube_ads',
    'basic_stats',
    'whitelist'
  ],

  PREMIUM_FEATURES: [
    'all_themes',           // 10+ themes (free gets 3)
    'sponsorblock',         // SponsorBlock integration
    'malware_protection',   // Real-time threat blocking
    'phishing_protection',  // Phishing site warnings
    'element_picker',       // Visual element blocker
    'custom_rules',         // Custom filter rules
    'advanced_filters',     // Additional filter lists
    'stats_history',        // Historical statistics
    'stats_export',         // Export stats as CSV/JSON
    'fingerprint_protection', // Anti-fingerprinting
    'priority_support'      // Priority customer support
  ],

  FREE_THEMES: ['cyber-blue', 'matrix-green', 'pure-light'],
  PREMIUM_THEMES: ['neon-purple', 'sunset-orange', 'midnight-ocean', 'rose-gold', 'arctic-frost', 'forest-emerald', 'dracula'],

  /**
   * Initialize premium system
   */
  async init() {
    await this.loadLicense();
    this.setupStorageListener();
    console.log('StreamShield Premium initialized. Premium:', this.isPremium);
    return this.isPremium;
  },

  /**
   * Load license from storage
   */
  async loadLicense() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['streamshieldLicense'], (result) => {
        if (result.streamshieldLicense) {
          const license = result.streamshieldLicense;
          this.licenseKey = license.key;
          this.expiresAt = license.expiresAt;
          this.lastValidatedAt = license.lastValidatedAt;

          // Check if still valid
          if (license.expiresAt && new Date(license.expiresAt) > new Date()) {
            this.isPremium = true;
          } else if (license.lifetime) {
            this.isPremium = true;
          } else {
            this.isPremium = false;
          }
        }
        resolve(this.isPremium);
      });
    });
  },

  /**
   * Validate a license key
   */
  async validateLicense(key) {
    try {
      // Try to validate with local proxy/desktop app first
      const response = await fetch('http://127.0.0.1:8081/api/license/validate', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ license_key: key })
      });

      if (response.ok) {
        const data = await response.json();
        if (data.valid) {
          const license = {
            key: key,
            validatedAt: new Date().toISOString(),
            lastValidatedAt: new Date().toISOString(),
            expiresAt: data.expires_at,
            lifetime: data.lifetime || false,
            tier: data.tier || 'premium'
          };

          await this.saveLicense(license);
          this.isPremium = true;
          return { success: true, license };
        }
      }

      return { success: false, error: 'Invalid license key' };
    } catch (e) {
      // Fallback: Accept certain key patterns for offline validation
      if (this.offlineValidate(key)) {
        const license = {
          key: key,
          validatedAt: new Date().toISOString(),
          lastValidatedAt: new Date().toISOString(),
          expiresAt: new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString(), // 1 year
          lifetime: false,
          tier: 'premium',
          offlineValidated: true
        };

        await this.saveLicense(license);
        this.isPremium = true;
        return { success: true, license };
      }

      return { success: false, error: 'Could not validate license: ' + e.message };
    }
  },

  /**
   * Offline validation for specific key patterns
   */
  offlineValidate(key) {
    // Accept keys that match pattern: SS-XXXX-XXXX-XXXX-XXXX
    const pattern = /^SS-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
    return pattern.test(key);
  },

  /**
   * Save license to storage
   */
  async saveLicense(license) {
    return new Promise((resolve) => {
      chrome.storage.sync.set({ streamshieldLicense: license }, resolve);
    });
  },

  /**
   * Remove license
   */
  async removeLicense() {
    this.isPremium = false;
    this.licenseKey = null;
    this.expiresAt = null;
    return new Promise((resolve) => {
      chrome.storage.sync.remove(['streamshieldLicense'], resolve);
    });
  },

  /**
   * Check if a feature is available
   */
  hasFeature(feature) {
    if (this.FREE_FEATURES.includes(feature)) {
      return true;
    }
    if (this.PREMIUM_FEATURES.includes(feature)) {
      return this.isPremium;
    }
    return false;
  },

  /**
   * Check if a theme is available
   */
  hasTheme(themeName) {
    if (this.FREE_THEMES.includes(themeName)) {
      return true;
    }
    if (this.PREMIUM_THEMES.includes(themeName)) {
      return this.isPremium;
    }
    return false;
  },

  /**
   * Get all available themes
   */
  getAvailableThemes() {
    if (this.isPremium) {
      return [...this.FREE_THEMES, ...this.PREMIUM_THEMES];
    }
    return this.FREE_THEMES;
  },

  /**
   * Listen for storage changes
   */
  setupStorageListener() {
    chrome.storage.onChanged.addListener((changes, area) => {
      if (area === 'sync' && changes.streamshieldLicense) {
        this.loadLicense();
      }
    });
  },

  /**
   * Get premium status summary
   */
  getStatus() {
    return {
      isPremium: this.isPremium,
      licenseKey: this.licenseKey ? this.maskKey(this.licenseKey) : null,
      expiresAt: this.expiresAt,
      lastValidatedAt: this.lastValidatedAt,
      availableFeatures: this.isPremium
        ? [...this.FREE_FEATURES, ...this.PREMIUM_FEATURES]
        : this.FREE_FEATURES,
      availableThemes: this.getAvailableThemes()
    };
  },

  /**
   * Mask license key for display
   */
  maskKey(key) {
    if (!key || key.length < 8) return '****';
    return key.substring(0, 3) + '-****-****-' + key.substring(key.length - 4);
  },

  /**
   * Show upgrade prompt
   */
  showUpgradePrompt(feature) {
    const featureNames = {
      'all_themes': 'Premium Themes',
      'sponsorblock': 'SponsorBlock',
      'malware_protection': 'Malware Protection',
      'phishing_protection': 'Phishing Protection',
      'element_picker': 'Element Picker',
      'custom_rules': 'Custom Rules',
      'advanced_filters': 'Advanced Filters',
      'stats_history': 'Statistics History',
      'stats_export': 'Export Statistics',
      'fingerprint_protection': 'Fingerprint Protection'
    };

    return {
      title: 'Premium Feature',
      message: `${featureNames[feature] || feature} is a premium feature. Upgrade to unlock all premium features!`,
      upgradeUrl: 'https://streamshield.io/premium',
      features: [
        '10+ Beautiful Themes',
        'SponsorBlock Integration',
        'Malware & Phishing Protection',
        'Element Picker',
        'Custom Filter Rules',
        'Advanced Statistics',
        'Priority Support'
      ]
    };
  }
};

// ============================================================================
// DESKTOP APP CONNECTION
// ============================================================================

const DesktopAppConnection = {
  isConnected: false,
  lastSyncTime: null,
  reconnectAttempts: 0,
  maxReconnectAttempts: 10,
  baseReconnectDelay: 2000, // 2 seconds base delay
  maxReconnectDelay: 60000, // 60 seconds max delay
  reconnectTimer: null,
  connectionListeners: [],
  lastError: null,
  connectionState: 'disconnected', // 'disconnected', 'connecting', 'connected', 'reconnecting'

  /**
   * Register a connection state listener
   */
  addConnectionListener(callback) {
    this.connectionListeners.push(callback);
  },

  /**
   * Notify all listeners of connection state change
   */
  notifyListeners() {
    const state = {
      isConnected: this.isConnected,
      connectionState: this.connectionState,
      lastSyncTime: this.lastSyncTime,
      reconnectAttempts: this.reconnectAttempts,
      lastError: this.lastError
    };
    this.connectionListeners.forEach(cb => {
      try { cb(state); } catch (e) { console.debug('Listener error:', e); }
    });
  },

  /**
   * Calculate exponential backoff delay
   */
  getReconnectDelay() {
    const delay = Math.min(
      this.baseReconnectDelay * Math.pow(2, this.reconnectAttempts),
      this.maxReconnectDelay
    );
    // Add jitter (0-25% random variation)
    return delay + Math.random() * delay * 0.25;
  },

  /**
   * Attempt to connect to the desktop app
   */
  async connect() {
    if (this.connectionState === 'connecting') {
      return { success: false, error: 'Connection already in progress' };
    }

    this.connectionState = 'connecting';
    this.notifyListeners();

    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 5000);

      const response = await fetch('http://127.0.0.1:8081/api/extension/handshake', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          extensionVersion: '4.0.0',
          timestamp: new Date().toISOString()
        }),
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      if (response.ok) {
        const data = await response.json();
        this.isConnected = true;
        this.connectionState = 'connected';
        this.lastSyncTime = new Date().toISOString();
        this.reconnectAttempts = 0;
        this.lastError = null;
        this.notifyListeners();
        console.log('Connected to StreamShield Desktop App:', data);

        // Start health check polling
        this.startHealthCheck();

        return { success: true, data };
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
    } catch (e) {
      this.isConnected = false;
      this.connectionState = 'disconnected';
      this.lastError = e.message;
      this.notifyListeners();
      console.debug('Desktop app not available:', e.message);
      return { success: false, error: e.message };
    }
  },

  /**
   * Start automatic reconnection with exponential backoff
   */
  startAutoReconnect() {
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
    }

    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      console.log('Max reconnect attempts reached, giving up');
      this.connectionState = 'disconnected';
      this.notifyListeners();
      return;
    }

    this.connectionState = 'reconnecting';
    this.notifyListeners();

    const delay = this.getReconnectDelay();
    console.log(`Reconnecting in ${Math.round(delay/1000)}s (attempt ${this.reconnectAttempts + 1}/${this.maxReconnectAttempts})`);

    this.reconnectTimer = setTimeout(async () => {
      this.reconnectAttempts++;
      const result = await this.connect();

      if (!result.success) {
        this.startAutoReconnect();
      }
    }, delay);
  },

  /**
   * Stop auto-reconnect attempts
   */
  stopAutoReconnect() {
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
      this.reconnectTimer = null;
    }
    this.reconnectAttempts = 0;
  },

  /**
   * Reset reconnect attempts (call when user manually reconnects)
   */
  resetReconnectAttempts() {
    this.reconnectAttempts = 0;
    this.stopAutoReconnect();
  },

  /**
   * Start periodic health check
   */
  startHealthCheck() {
    // Use alarm for health checks (more reliable than setInterval in service workers)
    chrome.alarms.create('desktopHealthCheck', { periodInMinutes: 0.5 }); // Every 30 seconds
  },

  /**
   * Perform health check
   */
  async healthCheck() {
    if (!this.isConnected) return;

    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000);

      const response = await fetch('http://127.0.0.1:8081/api/status', {
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      if (!response.ok) {
        throw new Error('Health check failed');
      }
    } catch (e) {
      console.log('Desktop app health check failed, attempting reconnect');
      this.isConnected = false;
      this.connectionState = 'disconnected';
      this.lastError = 'Connection lost';
      this.notifyListeners();
      this.startAutoReconnect();
    }
  },

  /**
   * Sync license status with desktop app
   */
  async syncLicenseStatus() {
    if (!this.isConnected) {
      const connectResult = await this.connect();
      if (!connectResult.success) return { success: false };
    }

    try {
      const response = await fetch('http://127.0.0.1:8081/api/license/status');
      if (response.ok) {
        const data = await response.json();

        // Update local premium status if desktop has valid license
        if (data.valid && data.license_key) {
          await StreamShieldPremium.validateLicense(data.license_key);
        }

        return { success: true, data };
      }
    } catch (e) {
      console.debug('Failed to sync license with desktop:', e.message);
    }
    return { success: false };
  },

  /**
   * Send blocking stats to desktop app
   */
  async sendStats(stats) {
    if (!this.isConnected) return { success: false };

    try {
      const response = await fetch('http://127.0.0.1:8081/api/extension/stats', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          stats: stats,
          timestamp: new Date().toISOString()
        })
      });

      if (response.ok) {
        return { success: true };
      }
    } catch (e) {
      console.debug('Failed to send stats to desktop:', e.message);
    }
    return { success: false };
  },

  /**
   * Get connection status
   */
  getStatus() {
    return {
      isConnected: this.isConnected,
      lastSyncTime: this.lastSyncTime
    };
  }
};

// ============================================================================
// CUSTOM BLOCKING RULES
// ============================================================================

let customBlockingRules = [];

/**
 * Load custom blocking rules from storage
 */
async function loadCustomRules() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['customBlockingRules'], (result) => {
      customBlockingRules = result.customBlockingRules || [];
      console.log('Loaded', customBlockingRules.length, 'custom blocking rules');
      resolve(customBlockingRules);
    });
  });
}

/**
 * Add a custom blocking rule
 */
async function addCustomRule(rule) {
  return new Promise((resolve) => {
    chrome.storage.local.get(['customBlockingRules'], (result) => {
      const rules = result.customBlockingRules || [];

      // Validate and add the rule
      const newRule = {
        id: Date.now(),
        pattern: rule.pattern,
        type: rule.type || 'domain', // domain, url, selector
        domain: rule.domain || '*',
        enabled: true,
        createdAt: new Date().toISOString()
      };

      rules.push(newRule);
      customBlockingRules = rules;

      chrome.storage.local.set({ customBlockingRules: rules }, () => {
        console.log('Added custom rule:', newRule);
        // Update declarativeNetRequest rules if needed
        updateBlockingRules();
        resolve({ success: true, rule: newRule });
      });
    });
  });
}

/**
 * Remove a custom blocking rule
 */
async function removeCustomRule(ruleId) {
  return new Promise((resolve) => {
    chrome.storage.local.get(['customBlockingRules'], (result) => {
      let rules = result.customBlockingRules || [];
      rules = rules.filter(r => r.id !== ruleId);
      customBlockingRules = rules;

      chrome.storage.local.set({ customBlockingRules: rules }, () => {
        console.log('Removed custom rule:', ruleId);
        updateBlockingRules();
        resolve({ success: true });
      });
    });
  });
}

/**
 * Get all custom blocking rules
 */
function getCustomRules() {
  return customBlockingRules;
}

// Configuration
const CONFIG = {
  proxyHost: 'localhost',
  proxyPort: 8080,
  dashboardUrl: 'http://localhost:8081',
  enableProxy: true,
  enableLocalBlocking: true,
  syncWithProxy: true,
  blockRedirects: true,
  blockPopups: true,
  enableMalwareProtection: true,
  enablePhishingProtection: true
};

// Load config from storage
chrome.storage.sync.get(['streamshieldConfig'], (result) => {
  if (result.streamshieldConfig) {
    CONFIG.blockRedirects = result.streamshieldConfig.blockRedirects !== false;
    CONFIG.blockPopups = result.streamshieldConfig.blockPopups !== false;
    CONFIG.enableMalwareProtection = result.streamshieldConfig.enableMalwareProtection !== false;
    CONFIG.enablePhishingProtection = result.streamshieldConfig.enablePhishingProtection !== false;
  }
});

// Statistics
let stats = {
  requestsBlocked: 0,
  adsBlocked: 0,
  trackersBlocked: 0,
  minersBlocked: 0,
  fingerprintingBlocked: 0,
  socialWidgetsBlocked: 0,
  pixelsBlocked: 0,
  threatsBlocked: 0,
  malwareBlocked: 0,
  phishingBlocked: 0,
  bytesBlocked: 0,
  dataSaved: 0,
  ytAdsSkipped: 0,
  ytSponsorsSkipped: 0,
  ytTimeSaved: 0
};

// Daily stats tracking
let dailyStats = {};

// Load persisted stats on startup
chrome.storage.local.get(['streamshieldStats', 'dailyStats'], (result) => {
  if (result.streamshieldStats) {
    stats = { ...stats, ...result.streamshieldStats };
    console.log('Loaded persisted stats:', stats);
  }
  if (result.dailyStats) {
    dailyStats = result.dailyStats;
    console.log('Loaded daily stats:', Object.keys(dailyStats).length, 'days');
  }
});

// Persist stats to storage periodically
let statsSaveTimer = null;
function persistStats() {
  if (statsSaveTimer) return; // Debounce
  statsSaveTimer = setTimeout(() => {
    chrome.storage.local.set({
      streamshieldStats: stats,
      dailyStats: dailyStats
    });
    statsSaveTimer = null;
    // Update badge when stats change
    updateBadge();
  }, 1000); // Save at most once per second
}

// ============================================================================
// BADGE UPDATE
// ============================================================================

/**
 * Update the extension badge with block count
 */
function updateBadge() {
  const totalBlocked = stats.adsBlocked + stats.trackersBlocked + stats.threatsBlocked;

  // Format the number for display
  let badgeText = '';
  if (totalBlocked >= 1000000) {
    badgeText = Math.floor(totalBlocked / 1000000) + 'M';
  } else if (totalBlocked >= 1000) {
    badgeText = Math.floor(totalBlocked / 1000) + 'K';
  } else if (totalBlocked > 0) {
    badgeText = totalBlocked.toString();
  }

  // Set badge text
  chrome.action.setBadgeText({ text: badgeText });

  // Set badge color based on premium status
  if (StreamShieldPremium.isPremium) {
    chrome.action.setBadgeBackgroundColor({ color: '#00C853' }); // Green for premium
  } else {
    chrome.action.setBadgeBackgroundColor({ color: '#2196F3' }); // Blue for free
  }
}

// Get today's date key for daily stats
function getTodayKey() {
  const today = new Date();
  return `${today.getFullYear()}-${String(today.getMonth() + 1).padStart(2, '0')}-${String(today.getDate()).padStart(2, '0')}`;
}

// Update daily stats
function updateDailyStats(category, count = 1) {
  const todayKey = getTodayKey();
  if (!dailyStats[todayKey]) {
    dailyStats[todayKey] = {
      ads: 0,
      trackers: 0,
      threats: 0,
      malware: 0,
      phishing: 0,
      total: 0
    };
  }
  dailyStats[todayKey][category] = (dailyStats[todayKey][category] || 0) + count;
  dailyStats[todayKey].total += count;

  // Clean up old stats (keep last 30 days)
  const keys = Object.keys(dailyStats).sort();
  while (keys.length > 30) {
    delete dailyStats[keys.shift()];
  }

  persistStats();
}

// Quick domain blacklist (synced from proxy)
let blacklist = new Set();
let whitelist = new Set();

// ============================================================================
// THREAT PROTECTION - Malware & Phishing Lists (v4.0 Feature)
// ============================================================================

// Known malware domains - loaded from storage or fetched
let malwareDomains = new Set();
let phishingDomains = new Set();
let threatListsLastUpdated = 0;
const THREAT_LIST_UPDATE_INTERVAL = 24 * 60 * 60 * 1000; // 24 hours

// Default threat domains (fallback if lists fail to load)
const DEFAULT_MALWARE_DOMAINS = new Set([
  'malware-site.example',
  'trojan-download.example',
  'ransomware.example',
  'keylogger-installer.example'
]);

const DEFAULT_PHISHING_DOMAINS = new Set([
  'paypa1-secure.example',
  'amaz0n-verify.example',
  'bankofamerica-login.example',
  'apple-id-suspended.example',
  'microsoft-account-verify.example'
]);

// Load threat lists from storage
async function loadThreatLists() {
  try {
    const data = await chrome.storage.local.get(['malwareDomains', 'phishingDomains', 'threatListsLastUpdated']);

    if (data.malwareDomains && Array.isArray(data.malwareDomains)) {
      malwareDomains = new Set(data.malwareDomains);
    } else {
      malwareDomains = new Set(DEFAULT_MALWARE_DOMAINS);
    }

    if (data.phishingDomains && Array.isArray(data.phishingDomains)) {
      phishingDomains = new Set(data.phishingDomains);
    } else {
      phishingDomains = new Set(DEFAULT_PHISHING_DOMAINS);
    }

    threatListsLastUpdated = data.threatListsLastUpdated || 0;

    console.log(`Loaded threat lists: ${malwareDomains.size} malware, ${phishingDomains.size} phishing domains`);

    // Check if we need to update
    if (Date.now() - threatListsLastUpdated > THREAT_LIST_UPDATE_INTERVAL) {
      fetchThreatLists();
    }
  } catch (error) {
    console.warn('Failed to load threat lists:', error);
    malwareDomains = new Set(DEFAULT_MALWARE_DOMAINS);
    phishingDomains = new Set(DEFAULT_PHISHING_DOMAINS);
  }
}

// Fetch threat lists from known sources
async function fetchThreatLists() {
  try {
    // Try to fetch from StreamShield proxy first
    const response = await fetch(`${CONFIG.dashboardUrl}/api/threats/lists`, {
      timeout: 10000
    });

    if (response.ok) {
      const data = await response.json();

      if (data.malware && Array.isArray(data.malware)) {
        malwareDomains = new Set(data.malware);
      }
      if (data.phishing && Array.isArray(data.phishing)) {
        phishingDomains = new Set(data.phishing);
      }

      // Save to storage
      await chrome.storage.local.set({
        malwareDomains: Array.from(malwareDomains),
        phishingDomains: Array.from(phishingDomains),
        threatListsLastUpdated: Date.now()
      });

      console.log(`Updated threat lists: ${malwareDomains.size} malware, ${phishingDomains.size} phishing domains`);
    }
  } catch (error) {
    console.warn('Failed to fetch threat lists from proxy:', error.message);
    // Fallback: keep existing lists or use defaults
  }
}

// Check if URL is a known threat
function checkThreat(url, domain) {
  // Extract base domain for comparison
  const parts = domain.split('.');

  // Check malware domains
  if (CONFIG.enableMalwareProtection) {
    if (malwareDomains.has(domain)) {
      return { type: 'malware', domain };
    }
    // Check parent domains
    for (let i = 0; i < parts.length - 1; i++) {
      const parentDomain = parts.slice(i).join('.');
      if (malwareDomains.has(parentDomain)) {
        return { type: 'malware', domain: parentDomain };
      }
    }
  }

  // Check phishing domains
  if (CONFIG.enablePhishingProtection) {
    if (phishingDomains.has(domain)) {
      return { type: 'phishing', domain };
    }
    // Check parent domains
    for (let i = 0; i < parts.length - 1; i++) {
      const parentDomain = parts.slice(i).join('.');
      if (phishingDomains.has(parentDomain)) {
        return { type: 'phishing', domain: parentDomain };
      }
    }
  }

  return null;
}

// ============================================================================
// Navigation Blocking for Threats (v4.0 Feature)
// ============================================================================

// Block navigation to malicious URLs
chrome.webNavigation.onBeforeNavigate.addListener(async (details) => {
  // Only check main frame navigations
  if (details.frameId !== 0) return;

  try {
    const url = new URL(details.url);
    const domain = url.hostname.toLowerCase();

    // Skip whitelisted domains
    if (isWhitelisted(domain)) return;

    // Check if it's a threat
    const threat = checkThreat(url, domain);

    if (threat) {
      // Update stats
      stats.threatsBlocked++;
      if (threat.type === 'malware') {
        stats.malwareBlocked++;
        updateDailyStats('malware');
      } else if (threat.type === 'phishing') {
        stats.phishingBlocked++;
        updateDailyStats('phishing');
      }
      updateDailyStats('threats');
      persistStats();

      console.log(`🚨 Blocked ${threat.type}: ${domain}`);

      // Redirect to warning page
      const warningUrl = chrome.runtime.getURL('warning.html') +
        `?type=${threat.type}&domain=${encodeURIComponent(domain)}&originalUrl=${encodeURIComponent(details.url)}`;

      chrome.tabs.update(details.tabId, { url: warningUrl });
    }
  } catch (error) {
    console.warn('Error checking navigation:', error);
  }
});

// ============================================================================
// SponsorBlock Settings (v4.0 Feature)
// ============================================================================

// Default SponsorBlock settings
let sponsorBlockSettings = {
  enabled: true,
  categories: {
    sponsor: true,
    selfpromo: true,
    interaction: true,
    intro: false,
    outro: false,
    preview: false,
    music_offtopic: false,
    filler: false
  },
  skipBehavior: 'auto', // 'auto', 'manual', 'off'
  showNotifications: true
};

// Load SponsorBlock settings
chrome.storage.sync.get(['sponsorBlockSettings'], (result) => {
  if (result.sponsorBlockSettings) {
    sponsorBlockSettings = { ...sponsorBlockSettings, ...result.sponsorBlockSettings };
  }
});

// ============================================================================
// Theme Support (v4.0 Feature)
// ============================================================================

let currentTheme = 'system'; // 'light', 'dark', 'system'

// Load theme preference
chrome.storage.sync.get(['theme'], (result) => {
  if (result.theme) {
    currentTheme = result.theme;
  }
});

// Broadcast theme changes to popup if open
function broadcastThemeChange(theme) {
  chrome.runtime.sendMessage({
    action: 'themeChanged',
    theme: theme
  }).catch(() => {
    // Popup not open, ignore
  });
}

// ============================================================================
// EXISTING FUNCTIONALITY (Preserved from v3.0)
// ============================================================================

// Hardcoded ad/redirect domains that should always be blocked
const BLOCKED_DOMAINS = new Set([
  // General ad networks
  'adsblocked.app',
  'adblocked.app',
  'ad-blocker.app',
  'popads.net',
  'popcash.net',
  'propellerads.com',
  'exoclick.com',
  'juicyads.com',
  'trafficjunky.com',
  'adcash.com',
  'admaven.com',
  'hilltopads.net',
  'clickadu.com',
  'revcontent.com',
  'mgid.com',
  'outbrain.com',
  'taboola.com',
  'zedo.com',
  'adform.net',
  'adnxs.com',
  'rubiconproject.com',
  'pubmatic.com',
  'openx.net',
  'criteo.com',
  'casalemedia.com',
  'advertising.com',
  // Streaming site ad networks (zoechip, gomovies, etc)
  // Only block ad/tracking NETWORKS, not content sites
  'betterads.co',
  'streamads.net',
  'onclickads.net',
  'onclickmax.com',
  'onclickmega.com',
  'clicksgear.com',
  'bidvertiser.com',
  'adsterra.com',
  'a-ads.com',
  'ad4game.com',
  'adxpremium.com',
  'adxpremium.services',
  'adf.ly',
  'bc.vc',
  'shorte.st',
  'sh.st',
  'linkbucks.com',
  'adlinkfly.link',
  'ouo.io',
  'ouo.press',
  'za.gl',
  'highperformanceformat.com',
  'highperformancecpm.com',
  'highperformancedisplayformat.com',
  'hrtya.com',
  'hrtyb.com',
  'hrtye.com',
  'ladsads.com',
  'landingtrackone.com',
  'newpopunder.com',
  'popmyads.com',
  'popunderjs.com',
  'popupads.ir',
  'revenuehits.com',
  'richpush.co',
  'push-ad.net',
  'push.express',
  'webpusher.net',
  'notix.io',
  'pushprofit.net',
  'pushsub.com',
  'pushance.com',
  'subscribstar.com',
  'voozmedia.com',
  'acint.net',
  'trafficforce.com'
]);

// NOTE: We block ad NETWORKS, not destination sites.
// Users can visit any site they want - we just block the
// malicious redirects/popups trying to hijack their clicks.

// VIDEO STREAMING WHITELIST - NEVER block these domains
// These are critical for video playback on anime/streaming sites
// IMPORTANT: Keep this list comprehensive to avoid rate limiting and video issues
const VIDEO_STREAMING_WHITELIST = new Set([
  // AnimeKai / 9anime / Zoro CDNs - ULTRA PASSIVE (all mirrors)
  'animekai.to', 'animekai.bz', 'animekai.tv', 'animekai.cc', 'animekai.lol',
  'animekai.ac', 'animekai.me', 'animekai.fit', 'animekai.org', 'animekai.net',
  'anikai.to', 'anikai.com', 'anikai.tv', 'anikai.cam', 'anikai.cc',
  'aniwave.to', 'aniwave.se', 'aniwave.vc', 'aniwave.bz',
  '9anime.to', '9anime.me', '9anime.id', '9anime.vc', '9animetv.to',
  'zoro.to', 'zoro.se', 'zoro.vc',
  'hianime.to', 'hianime.nz', 'hianime.sx',
  'aniwatch.to', 'aniwatch.me',
  // MegaCloud family (PRIMARY AnimeKai video host - CRITICAL)
  'megacloud.tv', 'megacloud.club', 'megaf.cc', 'megacloud.xyz',
  // RabbitStream / Vidplay (primary anime CDNs - CRITICAL)
  'rabbitstream.net', 'rapid-cloud.co', 'rapid-cloud.ru', 'rapid-cloud.live',
  'vidplay.site', 'vidplay.online', 'vidplay.lol', 'vidplay.tv',
  'vid142.site',
  'mcloud.to', 'mcloud.bz', 'mcloud2.to',
  // VidBinge / VidCloud family
  'vidbinge.com', 'vidcloud.co', 'vidcloud.online', 'vidcloud9.com',
  'vidstream.pro', 'vidstream.co',
  // VidSrc family
  'vidsrc.me', 'vidsrc.to', 'vidsrc.xyz', 'vidsrc.cc', 'vidsrc.pro',
  // SolarCDN (CRITICAL for AnimeKai)
  'solarcdn.me', 'solarcdn.cc', 'solarcdn.to', 'solarcdn.net',
  // GoGoAnime CDNs
  'gogostream.com', 'gogoanime.llc', 'gogoanime.tel', 'gogoanime.gr', 'gogoanime.gg',
  'gogocdn.net', 'anicdn.stream', 'anicache.net',
  'gogohd.net', 'gogohd.pro', 'streamani.net',
  'streamsb.net', 'sbplay.one', 'sbplay2.xyz',
  // Anitaku (GoGoAnime rebrand)
  'anitaku.to', 'anitaku.so', 'anitaku.bz',
  // DokiCloud
  'dokicloud.one', 'dokicloud.net',
  // Common anime player domains
  'kwik.cx', 'kwik.si', 'kwik.to',
  'streamtape.com', 'streamtape.net', 'streamtape.xyz',
  'streamlare.com',
  'mixdrop.co', 'mixdrop.to', 'mixdrop.sx', 'mixdrop.ag',
  'filemoon.sx', 'filemoon.to', 'filemoon.in',
  'streamwish.to', 'streamwish.com', 'streamwish.site',
  'mp4upload.com',
  'doodstream.com', 'dood.watch', 'dood.to', 'dood.so', 'dood.la',
  // AllAnime CDN
  'allanime.day', 'allanime.to', 'blog.allanime.day',
  // General anime/streaming infrastructure
  'animepahe.com', 'animepahe.ru',
  'binanime.com', 'consumet.org',
  // HLS/Video infrastructure
  'cdn.plyr.io', 'cdn.jwplayer.com',
  'content.jwplatform.com', 'ssl.p.jwpcdn.com',
  'cdn.flowplayer.com', 'cdn.vidible.tv', 'cdn.video.akamai.com',
  // Streaming embeds
  'fembed.com', 'femax20.com', 'fcdn.stream',
  'embedsito.com', 'eplayvid.net',
  // Crunchyroll / Funimation
  'crunchyroll.com', 'funimation.com',
  'pl.crunchyroll.com', 'v.vrv.co',
  // ZoeChip / GoMovies / Solarmovie / FMovies
  'zoechip.com', 'zoechip.cc', 'zoechip.gg',
  'gomovies.sx', 'solarmovie.pe',
  'fmovies.to', 'fmovies.wtf', 'fmovies.name', 'fmovies.llc',
  '123movies.ai'
]);

// Tracker domains - analytics/tracking that don't break sites
const TRACKER_DOMAINS = new Set([
  // Google Analytics/Tracking (blocks tracking, not essential Google services)
  'google-analytics.com',
  'googletagmanager.com',
  'googletagservices.com',
  'pagead2.googlesyndication.com',
  'googleadservices.com',
  'stats.g.doubleclick.net',
  // Facebook/Meta Tracking
  'connect.facebook.net',
  'pixel.facebook.com',
  'graph.facebook.com',
  // Microsoft/Bing
  'bat.bing.com',
  'clarity.ms',
  // Other major trackers
  'hotjar.com',
  'mouseflow.com',
  'fullstory.com',
  'amplitude.com',
  'segment.com',
  'segment.io',
  'mixpanel.com',
  'heap.io',
  'heapanalytics.com',
  'kissmetrics.com',
  'quantserve.com',
  'scorecardresearch.com',
  'omtrdc.net',
  'demdex.net',
  'bluekai.com',
  'krxd.net',
  'adsrvr.org',
  'thetradedesk.com',
  'liveramp.com',
  'rlcdn.com',
  'pippio.com',
  'liadm.com',
  'mxpnl.com',
  'nr-data.net',
  'newrelic.com',
  'sentry.io',
  'bugsnag.com',
  'loggly.com',
  'splunk.com',
  // Social trackers
  'platform.twitter.com',
  'analytics.twitter.com',
  'cdn.syndication.twimg.com',
  'platform.linkedin.com',
  'snap.licdn.com',
  'ads.linkedin.com',
  'px.ads.linkedin.com',
  'analytics.tiktok.com',
  'ads.pinterest.com',
  // Ad verification
  'moatads.com',
  'doubleverify.com',
  'adsafeprotected.com',
  'iasds01.com',
  // Fingerprinting services
  'iovation.com',
  'threatmetrix.com',
  'deviceprint.io',
  'fingerprintjs.com'
]);

// Initialize whitelist from both storage locations
async function initializeWhitelist() {
  try {
    // Get from both local and sync storage
    const [localData, syncData] = await Promise.all([
      chrome.storage.local.get(['whitelist']),
      chrome.storage.sync.get(['whitelist'])
    ]);

    // Merge whitelists from both sources
    const localWhitelist = localData.whitelist || [];
    const syncWhitelist = syncData.whitelist || [];
    const merged = [...new Set([...localWhitelist, ...syncWhitelist])];

    whitelist = new Set(merged);

    // Save merged list to both storages
    await Promise.all([
      chrome.storage.local.set({ whitelist: merged }),
      chrome.storage.sync.set({ whitelist: merged })
    ]);

    console.log(`Whitelist initialized: ${whitelist.size} domains`);
  } catch (error) {
    console.warn('Failed to initialize whitelist:', error);
  }
}

// Sync whitelist changes to proxy
async function syncWhitelistToProxy() {
  if (!CONFIG.syncWithProxy) return;

  try {
    const whitelistArray = Array.from(whitelist);
    const response = await fetch(`${CONFIG.dashboardUrl}/api/extension/config`, {
      method: 'PATCH',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ whitelist: whitelistArray })
    });

    if (response.ok) {
      console.log('Whitelist synced to proxy');
    }
  } catch (error) {
    // Proxy might not be running - that is okay
    console.debug('Proxy sync skipped (not running):', error.message);
  }
}

// Listen for storage changes to keep whitelist in sync
chrome.storage.onChanged.addListener((changes, areaName) => {
  if (changes.whitelist) {
    const newWhitelist = changes.whitelist.newValue || [];
    whitelist = new Set(newWhitelist);

    // Sync to the other storage area
    if (areaName === 'sync') {
      chrome.storage.local.set({ whitelist: newWhitelist });
    } else if (areaName === 'local') {
      chrome.storage.sync.set({ whitelist: newWhitelist });
    }

    // Sync to proxy
    syncWhitelistToProxy();

    console.log(`Whitelist updated from ${areaName}: ${whitelist.size} domains`);
  }

  // Handle theme changes (v4.0)
  if (changes.theme) {
    currentTheme = changes.theme.newValue;
    broadcastThemeChange(currentTheme);
  }

  // Handle SponsorBlock settings changes (v4.0)
  if (changes.sponsorBlockSettings) {
    sponsorBlockSettings = { ...sponsorBlockSettings, ...changes.sponsorBlockSettings.newValue };
    // Broadcast to content scripts
    chrome.tabs.query({ url: '*://*.youtube.com/*' }, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.sendMessage(tab.id, {
          action: 'sponsorBlockSettingsUpdated',
          settings: sponsorBlockSettings
        }).catch(() => {});
      });
    });
  }

  // Handle config changes
  if (changes.streamshieldConfig) {
    const newConfig = changes.streamshieldConfig.newValue || {};
    CONFIG.blockRedirects = newConfig.blockRedirects !== false;
    CONFIG.blockPopups = newConfig.blockPopups !== false;
    CONFIG.enableMalwareProtection = newConfig.enableMalwareProtection !== false;
    CONFIG.enablePhishingProtection = newConfig.enablePhishingProtection !== false;
  }
});

// Initialize on browser startup
chrome.runtime.onStartup.addListener(async () => {
  console.log('StreamShield v4.0.0 starting up...');

  // Initialize premium features
  await StreamShieldPremium.init();

  // Load custom blocking rules
  await loadCustomRules();

  initializeWhitelist();
  loadThreatLists();
  await updateBlockingRules();
  syncWithProxy();

  // Try to connect to desktop app
  DesktopAppConnection.connect().then(result => {
    if (result.success) {
      DesktopAppConnection.syncLicenseStatus();
    }
  });

  // Update badge with block count
  updateBadge();
});

// Initialize
chrome.runtime.onInstalled.addListener(async (details) => {
  console.log('StreamShield v4.0.0 installed!', details.reason);

  // Initialize premium features first
  await StreamShieldPremium.init();

  // Load custom blocking rules
  await loadCustomRules();

  // Load saved settings
  chrome.storage.local.get(['config', 'blacklist'], (data) => {
    if (data.config) {
      Object.assign(CONFIG, data.config);
    }
    if (data.blacklist) {
      blacklist = new Set(data.blacklist);
    }
  });

  // Initialize whitelist from both storage locations
  initializeWhitelist();

  // Load threat lists (v4.0)
  loadThreatLists();

  // Create blocking rules for Manifest V3
  await updateBlockingRules();

  // Sync with StreamShield proxy
  syncWithProxy();

  // Set up sync interval
  setInterval(syncWithProxy, 300000); // Every 5 minutes

  // Set up threat list refresh (v4.0)
  chrome.alarms.create('refreshThreatLists', { periodInMinutes: 60 * 6 }); // Every 6 hours

  // Set up periodic license validation (every 24 hours)
  chrome.alarms.create('validateLicense', { periodInMinutes: 60 * 24 }); // Every 24 hours

  // Set up stats sync with desktop app (every 5 minutes)
  chrome.alarms.create('syncDesktopStats', { periodInMinutes: 5 });

  // Try to connect to desktop app on install
  DesktopAppConnection.connect().then(result => {
    if (result.success) {
      DesktopAppConnection.syncLicenseStatus();
    }
  });

  // Create context menus
  createContextMenus();

  // Update badge with block count
  updateBadge();
});

// Handle alarms (v4.0)
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'refreshThreatLists') {
    fetchThreatLists();
  }
  else if (alarm.name === 'desktopHealthCheck') {
    DesktopAppConnection.healthCheck();
  }
  else if (alarm.name === 'validateLicense') {
    // Periodic license validation (every 24 hours)
    if (StreamShieldPremium.licenseKey) {
      StreamShieldPremium.validateLicense(StreamShieldPremium.licenseKey)
        .then(result => {
          console.log('Periodic license validation:', result.success ? 'valid' : 'invalid');
          if (!result.success) {
            // License is no longer valid - update badge
            updateBadge();
          }
        });
    }
  }
  else if (alarm.name === 'syncDesktopStats') {
    // Sync stats with desktop app
    if (DesktopAppConnection.isConnected) {
      DesktopAppConnection.sendStats(getStats());
    } else {
      // Try to reconnect
      DesktopAppConnection.connect().then(result => {
        if (result.success) {
          DesktopAppConnection.sendStats(getStats());
        }
      });
    }
  }
});

// Note: Request blocking is now handled by declarativeNetRequest rules
// The dynamic rules will be created from BLOCKED_DOMAINS and TRACKER_DOMAINS
// This allows Manifest V3 compatibility

// Create declarativeNetRequest rules on install/update
async function updateBlockingRules() {
  try {
    // Get current dynamic rules
    const existingRules = await chrome.declarativeNetRequest.getDynamicRules();
    const ruleIds = existingRules.map(rule => rule.id);

    // Remove old rules
    if (ruleIds.length > 0) {
      await chrome.declarativeNetRequest.updateDynamicRules({
        removeRuleIds: ruleIds
      });
    }

    // Create new rules from blocked domains
    const rules = [];
    let ruleId = 1;

    // Add rules for blocked domains
    const allBlockedDomains = [...BLOCKED_DOMAINS, ...TRACKER_DOMAINS];
    for (const domain of allBlockedDomains) {
      if (ruleId > 5000) break; // Chrome limit is 5000 dynamic rules

      rules.push({
        id: ruleId++,
        priority: 1,
        action: { type: "block" },
        condition: {
          urlFilter: `*://*.${domain}/*`,
          resourceTypes: ["main_frame", "sub_frame", "script", "image", "stylesheet", "xmlhttprequest"]
        }
      });
    }

    // Add rules to block
    if (rules.length > 0) {
      await chrome.declarativeNetRequest.updateDynamicRules({
        addRules: rules
      });
      console.log(`Created ${rules.length} blocking rules`);
    }
  } catch (error) {
    console.warn('Failed to update blocking rules:', error);
  }
}

// Track blocked requests (for stats) - Multiple methods for reliability
// Method 1: onRuleMatchedDebug (only works in dev mode with feedback permission)
if (chrome.declarativeNetRequest.onRuleMatchedDebug) {
  chrome.declarativeNetRequest.onRuleMatchedDebug.addListener((details) => {
    stats.requestsBlocked++;
    stats.adsBlocked++;
    stats.dataSaved += 50000; // Estimate 50KB per blocked request
    updateDailyStats('ads');
    persistStats();
    console.log(`Blocked: ${details.request.url}`);
  });
}

// Method 2: Use getMatchedRules to get session rule counts (more reliable)
async function updateStatsFromRules() {
  try {
    const matchedRules = await chrome.declarativeNetRequest.getMatchedRules();
    if (matchedRules && matchedRules.rulesMatchedInfo) {
      const ruleCount = matchedRules.rulesMatchedInfo.length;
      if (ruleCount > stats.requestsBlocked) {
        const newBlocks = ruleCount - stats.requestsBlocked;
        stats.requestsBlocked = ruleCount;
        stats.adsBlocked += newBlocks;
        stats.dataSaved += newBlocks * 50000;
        updateDailyStats('ads', newBlocks);
        persistStats();
      }
    }
  } catch (e) {
    // getMatchedRules may not be available in all contexts
  }
}

// Method 3: Periodic stats check alarm
chrome.alarms.create('updateBlockStats', { periodInMinutes: 0.1 }); // Every 6 seconds

// Listen for the alarm
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'updateBlockStats') {
    updateStatsFromRules();
  }
});

// Check if domain should be blocked
function shouldBlock(url, domain, type) {
  // Check hardcoded blocked domains first
  if (BLOCKED_DOMAINS.has(domain)) {
    stats.adsBlocked++;
    return true;
  }

  // Check parent domains against hardcoded list
  const parts = domain.split('.');
  for (let i = 0; i < parts.length - 1; i++) {
    const parentDomain = parts.slice(i).join('.');
    if (BLOCKED_DOMAINS.has(parentDomain)) {
      stats.adsBlocked++;
      return true;
    }
  }

  // Check tracker domains (privacy protection)
  if (TRACKER_DOMAINS.has(domain)) {
    stats.trackersBlocked++;
    return true;
  }

  // Check parent domains against tracker list
  for (let i = 0; i < parts.length - 1; i++) {
    const parentDomain = parts.slice(i).join('.');
    if (TRACKER_DOMAINS.has(parentDomain)) {
      stats.trackersBlocked++;
      return true;
    }
  }

  // Check local blacklist
  if (blacklist.has(domain)) {
    return true;
  }

  // Check parent domains against dynamic blacklist
  for (let i = 0; i < parts.length - 1; i++) {
    const parentDomain = parts.slice(i).join('.');
    if (blacklist.has(parentDomain)) {
      return true;
    }
  }

  // Check URL patterns
  const urlStr = url.href.toLowerCase();
  const pathname = url.pathname.toLowerCase();

  // Ad redirect patterns (like /rei.php, /go.php, /out.php, /click.php)
  if (CONFIG.blockRedirects && pathname.match(/^\/(rei|go|out|click|redir|redirect|track|pop|ad)\.(php|html?|aspx?)$/i)) {
    return true;
  }

  // Affiliate redirect patterns (click_id, exid, tracking params)
  if (CONFIG.blockRedirects) {
    const searchParams = url.search.toLowerCase();
    if (searchParams.match(/[?&](click_id|clickid|exid|aff_id|affid|ref_id|refid|campaign_id|pid|sid|tid)=/i)) {
      // Check if it looks like an ad redirect (not a legitimate site)
      if (pathname.match(/\/(display|affiliate|partner|click|track|out|go|redir)/i)) {
        return true;
      }
    }
  }

  // Block known affiliate/ad redirect URL patterns
  if (CONFIG.blockRedirects && pathname.match(/\/display\/(other|ad|banner|popup)/i)) {
    return true;
  }

  // Popup/popunder patterns
  if (CONFIG.blockPopups && urlStr.match(/popunder|popads|popup|clickunder/i)) {
    return true;
  }

  // Ad patterns
  if (urlStr.match(/\/ads?[\/\?]/i) ||
      urlStr.match(/[_.-]ad[s_.-]/i) ||
      urlStr.match(/doubleclick|googlesyndication|googleadservices/i)) {
    return true;
  }

  // Tracking patterns
  if (urlStr.match(/analytics|track|beacon|pixel/i) &&
      type === 'image' && urlStr.includes('.gif')) {
    return true;
  }

  // Crypto miner patterns
  if (urlStr.match(/coinhive|crypto-loot|jsecoin|webmine/i)) {
    return true;
  }

  // Fingerprinting patterns
  if (urlStr.match(/fingerprint|device-?print|iovation/i)) {
    return true;
  }

  return false;
}

// Check if domain is whitelisted
function isWhitelisted(domain) {
  // Check user whitelist first
  if (whitelist.has(domain)) {
    return true;
  }

  // Check hardcoded video streaming whitelist (critical for video playback)
  if (VIDEO_STREAMING_WHITELIST.has(domain)) {
    return true;
  }

  // Check parent domains against user whitelist
  const parts = domain.split('.');
  for (let i = 0; i < parts.length - 1; i++) {
    const parentDomain = parts.slice(i).join('.');
    if (whitelist.has(parentDomain)) {
      return true;
    }
    // Also check video streaming whitelist for parent domains
    if (VIDEO_STREAMING_WHITELIST.has(parentDomain)) {
      return true;
    }
  }

  return false;
}

// Update category statistics
function updateCategoryStats(url, domain) {
  const urlStr = url.href.toLowerCase();

  // Categorize the block
  if (urlStr.match(/\/ads?[\/\?]/i) || urlStr.match(/[_.-]ad[s_.-]/i)) {
    stats.adsBlocked++;
    updateDailyStats('ads');
  }
  else if (urlStr.match(/analytics|track|beacon/i)) {
    stats.trackersBlocked++;
    updateDailyStats('trackers');
  }
  else if (urlStr.match(/coinhive|crypto-loot|jsecoin/i)) {
    stats.minersBlocked++;
  }
  else if (urlStr.match(/fingerprint|device-?print/i)) {
    stats.fingerprintingBlocked++;
  }
  else if (urlStr.match(/facebook|twitter|linkedin|pinterest/i) &&
           urlStr.match(/share|like|follow|widget/i)) {
    stats.socialWidgetsBlocked++;
  }
  else if (url.pathname.match(/\.(gif|png)$/) && url.search) {
    stats.pixelsBlocked++;
  }
  persistStats();
}

// Estimate request size
function estimateRequestSize(details) {
  // Rough estimate based on request type
  const sizeMap = {
    'script': 50000,    // 50 KB
    'image': 20000,     // 20 KB
    'stylesheet': 15000, // 15 KB
    'media': 100000,    // 100 KB
    'font': 30000,      // 30 KB
    'other': 5000       // 5 KB
  };

  return sizeMap[details.type] || 5000;
}

// Sync with StreamShield proxy
async function syncWithProxy() {
  if (!CONFIG.syncWithProxy) {
    return;
  }

  try {
    // Get filter lists from proxy
    const response = await fetch(`${CONFIG.dashboardUrl}/api/filters/export`);

    if (response.ok) {
      const data = await response.json();

      // Update blacklist
      if (data.blacklist) {
        blacklist = new Set(data.blacklist);
        chrome.storage.local.set({ blacklist: Array.from(blacklist) });
      }

      // Update whitelist
      if (data.whitelist) {
        whitelist = new Set(data.whitelist);
        chrome.storage.local.set({ whitelist: Array.from(whitelist) });
      }

      console.log(`Synced with proxy: ${blacklist.size} blacklisted domains`);
    }
  } catch (error) {
    console.warn('Failed to sync with proxy:', error.message);
  }
}

// Get statistics
function getStats() {
  return {
    ...stats,
    blacklistSize: blacklist.size,
    whitelistSize: whitelist.size,
    megabytesSaved: (stats.bytesBlocked / 1024 / 1024).toFixed(2),
    // Alias fields for popup compatibility
    dataSaved: stats.dataSaved || stats.bytesBlocked,
    ytAdsSkipped: stats.ytAdsSkipped,
    ytSponsorsSkipped: stats.ytSponsorsSkipped,
    ytTimeSaved: stats.ytTimeSaved,
    // v4.0 fields
    threatsBlocked: stats.threatsBlocked,
    malwareBlocked: stats.malwareBlocked,
    phishingBlocked: stats.phishingBlocked
  };
}

// Get daily stats (v4.0)
function getDailyStats(days = 7) {
  const result = {};
  const keys = Object.keys(dailyStats).sort().slice(-days);
  keys.forEach(key => {
    result[key] = dailyStats[key];
  });
  return result;
}

// ============================================================================
// Element Picker Support (v4.0 Feature)
// ============================================================================

async function launchElementPicker(tabId) {
  try {
    // Inject the element picker script
    await chrome.scripting.executeScript({
      target: { tabId: tabId },
      files: ['element-picker.js']
    });

    // Send activation message to the content script
    // Wait a brief moment for the script to initialize
    setTimeout(() => {
      chrome.tabs.sendMessage(tabId, { action: 'activateElementPicker' })
        .catch(err => console.log('Element picker activation sent'));
    }, 100);

    console.log('Element picker injected into tab:', tabId);
    return { success: true };
  } catch (error) {
    console.error('Failed to inject element picker:', error);
    return { success: false, error: error.message };
  }
}

// ============================================================================
// Message Handler (Extended for v4.0)
// ============================================================================

// Track per-tab blocked counts from content scripts
const tabBlockedCounts = new Map();

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  // Existing handlers
  if (request.action === 'getStats') {
    sendResponse(getStats());
    return true;
  }
  // Handle pageInfo from content scripts - this is the primary way to track blocked items
  else if (request.action === 'pageInfo') {
    const tabId = sender.tab?.id;
    if (tabId && request.blocked > 0) {
      const previousCount = tabBlockedCounts.get(tabId) || 0;
      const newBlocks = request.blocked - previousCount;

      if (newBlocks > 0) {
        // Update our stats with the new blocks from this tab
        stats.adsBlocked += newBlocks;
        stats.requestsBlocked += newBlocks;
        stats.dataSaved += newBlocks * 25000; // Estimate 25KB per blocked element
        updateDailyStats('ads', newBlocks);
        persistStats();
        console.log(`Tab ${tabId}: +${newBlocks} blocks (total: ${request.blocked})`);
      }
      tabBlockedCounts.set(tabId, request.blocked);
    }
    sendResponse({ success: true });
    return true;
  }
  // Report blocked item from content script
  else if (request.action === 'blocked') {
    stats.adsBlocked++;
    stats.requestsBlocked++;
    stats.dataSaved += 25000;
    updateDailyStats('ads');
    persistStats();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'addToWhitelist') {
    whitelist.add(request.domain);
    const whitelistArray = Array.from(whitelist);
    // Save to both storages
    chrome.storage.local.set({ whitelist: whitelistArray });
    chrome.storage.sync.set({ whitelist: whitelistArray });
    // Sync to proxy
    syncWhitelistToProxy();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'removeFromWhitelist') {
    whitelist.delete(request.domain);
    const whitelistArray = Array.from(whitelist);
    // Save to both storages
    chrome.storage.local.set({ whitelist: whitelistArray });
    chrome.storage.sync.set({ whitelist: whitelistArray });
    // Sync to proxy
    syncWhitelistToProxy();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'getWhitelist') {
    sendResponse({ whitelist: Array.from(whitelist) });
    return true;
  }
  else if (request.action === 'sync') {
    syncWithProxy();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'updateConfig') {
    if (request.config) {
      CONFIG.blockRedirects = request.config.blockRedirects !== false;
      CONFIG.blockPopups = request.config.blockPopups !== false;
      CONFIG.enableMalwareProtection = request.config.enableMalwareProtection !== false;
      CONFIG.enablePhishingProtection = request.config.enablePhishingProtection !== false;
      // Save to storage
      chrome.storage.sync.set({ streamshieldConfig: request.config });
    }
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'ytAdSkipped') {
    stats.ytAdsSkipped++;
    stats.ytTimeSaved += (request.duration || 30); // Estimate 30s per ad
    persistStats();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'ytSponsorSkipped') {
    stats.ytSponsorsSkipped++;
    stats.ytTimeSaved += (request.duration || 60); // Estimate 60s per sponsor
    persistStats();
    sendResponse({ success: true });
    return true;
  }
  else if (request.action === 'updateStats') {
    // Allow content scripts to update stats directly
    if (request.stats) {
      Object.assign(stats, request.stats);
      persistStats();
    }
    sendResponse({ success: true });
    return true;
  }

  // ============================================================================
  // v4.0 Message Handlers
  // ============================================================================

  // Element Picker (v4.0)
  else if (request.action === 'launchElementPicker') {
    const tabId = request.tabId || (sender.tab && sender.tab.id);
    if (tabId) {
      launchElementPicker(tabId).then(sendResponse);
      return true; // Async response
    } else {
      // Get active tab if no tabId provided
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]) {
          launchElementPicker(tabs[0].id).then(sendResponse);
        } else {
          sendResponse({ success: false, error: 'No active tab found' });
        }
      });
      return true; // Async response
    }
  }

  // Add custom filter from element picker (v4.0)
  else if (request.action === 'addCustomFilter') {
    chrome.storage.local.get(['customFilters'], (data) => {
      const filters = data.customFilters || [];
      filters.push({
        selector: request.selector,
        domain: request.domain,
        createdAt: Date.now()
      });
      chrome.storage.local.set({ customFilters: filters });
      sendResponse({ success: true });
    });
    return true; // Async response
  }

  // Get daily stats (v4.0)
  else if (request.action === 'getDailyStats') {
    sendResponse({ stats: getDailyStats(request.days || 7) });
    return true;
  }

  // Get SponsorBlock settings (v4.0)
  else if (request.action === 'getSponsorBlockSettings') {
    sendResponse({ settings: sponsorBlockSettings });
    return true;
  }

  // Update SponsorBlock settings (v4.0)
  else if (request.action === 'updateSponsorBlockSettings') {
    if (request.settings) {
      sponsorBlockSettings = { ...sponsorBlockSettings, ...request.settings };
      chrome.storage.sync.set({ sponsorBlockSettings: sponsorBlockSettings });
      // Broadcast to content scripts on YouTube
      chrome.tabs.query({ url: '*://*.youtube.com/*' }, (tabs) => {
        tabs.forEach(tab => {
          chrome.tabs.sendMessage(tab.id, {
            action: 'sponsorBlockSettingsUpdated',
            settings: sponsorBlockSettings
          }).catch(() => {});
        });
      });
    }
    sendResponse({ success: true });
    return true;
  }

  // Get theme (v4.0)
  else if (request.action === 'getTheme') {
    sendResponse({ theme: currentTheme });
    return true;
  }

  // Set theme (v4.0)
  else if (request.action === 'setTheme') {
    currentTheme = request.theme;
    chrome.storage.sync.set({ theme: currentTheme });
    broadcastThemeChange(currentTheme);
    sendResponse({ success: true });
    return true;
  }

  // Get threat protection status (v4.0)
  else if (request.action === 'getThreatProtectionStatus') {
    sendResponse({
      malwareEnabled: CONFIG.enableMalwareProtection,
      phishingEnabled: CONFIG.enablePhishingProtection,
      malwareDomainsCount: malwareDomains.size,
      phishingDomainsCount: phishingDomains.size,
      lastUpdated: threatListsLastUpdated
    });
    return true;
  }

  // Force refresh threat lists (v4.0)
  else if (request.action === 'refreshThreatLists') {
    fetchThreatLists().then(() => {
      sendResponse({ success: true });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true; // Async response
  }

  // Proceed to blocked site (user override) (v4.0)
  else if (request.action === 'proceedToBlockedSite') {
    // Add to temporary session whitelist
    const domain = request.domain;
    if (domain) {
      // Add to session-only whitelist (not persisted)
      whitelist.add(domain);
      // Navigate to the original URL
      if (sender.tab && request.originalUrl) {
        chrome.tabs.update(sender.tab.id, { url: request.originalUrl });
      }
      sendResponse({ success: true });
    } else {
      sendResponse({ success: false, error: 'No domain provided' });
    }
    return true;
  }

  // ============================================================================
  // PREMIUM FEATURE MESSAGE HANDLERS
  // ============================================================================

  // Get premium status
  else if (request.action === 'getPremiumStatus') {
    sendResponse(StreamShieldPremium.getStatus());
    return true;
  }

  // Validate license key
  else if (request.action === 'validateLicense') {
    if (request.licenseKey) {
      StreamShieldPremium.validateLicense(request.licenseKey)
        .then(result => {
          // Update badge after license validation
          updateBadge();
          sendResponse(result);
        });
      return true; // Async response
    } else {
      sendResponse({ success: false, error: 'No license key provided' });
      return true;
    }
  }

  // Remove/deactivate license
  else if (request.action === 'removeLicense') {
    StreamShieldPremium.removeLicense()
      .then(() => {
        updateBadge();
        sendResponse({ success: true });
      });
    return true; // Async response
  }

  // Check if a specific feature is available
  else if (request.action === 'hasFeature') {
    const hasIt = StreamShieldPremium.hasFeature(request.feature);
    if (!hasIt && request.showUpgrade) {
      sendResponse({
        available: false,
        upgradePrompt: StreamShieldPremium.showUpgradePrompt(request.feature)
      });
    } else {
      sendResponse({ available: hasIt });
    }
    return true;
  }

  // Activate element picker (premium feature check)
  else if (request.action === 'activateElementPicker') {
    // Check if premium (element picker is a premium feature)
    if (!StreamShieldPremium.hasFeature('element_picker')) {
      sendResponse({
        success: false,
        requiresPremium: true,
        upgradePrompt: StreamShieldPremium.showUpgradePrompt('element_picker')
      });
      return true;
    }

    // Get the active tab and launch element picker
    const tabId = request.tabId || (sender.tab && sender.tab.id);
    if (tabId) {
      launchElementPicker(tabId).then(sendResponse);
    } else {
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]) {
          launchElementPicker(tabs[0].id).then(sendResponse);
        } else {
          sendResponse({ success: false, error: 'No active tab found' });
        }
      });
    }
    return true; // Async response
  }

  // Add custom blocking rule (premium feature)
  else if (request.action === 'addCustomRule') {
    if (!StreamShieldPremium.hasFeature('custom_rules')) {
      sendResponse({
        success: false,
        requiresPremium: true,
        upgradePrompt: StreamShieldPremium.showUpgradePrompt('custom_rules')
      });
      return true;
    }

    if (request.rule) {
      addCustomRule(request.rule).then(sendResponse);
      return true; // Async response
    } else {
      sendResponse({ success: false, error: 'No rule provided' });
      return true;
    }
  }

  // Remove custom blocking rule
  else if (request.action === 'removeCustomRule') {
    if (request.ruleId) {
      removeCustomRule(request.ruleId).then(sendResponse);
      return true; // Async response
    } else {
      sendResponse({ success: false, error: 'No rule ID provided' });
      return true;
    }
  }

  // Get all custom blocking rules
  else if (request.action === 'getCustomRules') {
    sendResponse({ rules: getCustomRules() });
    return true;
  }

  // Get SponsorBlock status (checks if enabled and premium)
  else if (request.action === 'getSponsorBlockStatus') {
    const isPremium = StreamShieldPremium.hasFeature('sponsorblock');
    sendResponse({
      enabled: isPremium && sponsorBlockSettings.enabled,
      isPremium: isPremium,
      settings: sponsorBlockSettings,
      requiresPremium: !isPremium ? StreamShieldPremium.showUpgradePrompt('sponsorblock') : null
    });
    return true;
  }

  // Get desktop app connection status
  else if (request.action === 'getDesktopConnectionStatus') {
    sendResponse({
      ...DesktopAppConnection.getStatus(),
      connectionState: DesktopAppConnection.connectionState,
      reconnectAttempts: DesktopAppConnection.reconnectAttempts,
      maxReconnectAttempts: DesktopAppConnection.maxReconnectAttempts,
      lastError: DesktopAppConnection.lastError
    });
    return true;
  }

  // Force reconnect to desktop app
  else if (request.action === 'forceReconnect') {
    DesktopAppConnection.resetReconnectAttempts();
    DesktopAppConnection.connect().then(result => {
      if (!result.success) {
        DesktopAppConnection.startAutoReconnect();
      }
      sendResponse(result);
    });
    return true; // Async response
  }

  // Connect to desktop app
  else if (request.action === 'connectToDesktopApp') {
    DesktopAppConnection.connect().then(result => {
      if (result.success) {
        DesktopAppConnection.syncLicenseStatus().then(() => {
          sendResponse(result);
        });
      } else {
        sendResponse(result);
      }
    });
    return true; // Async response
  }

  // Sync license with desktop app
  else if (request.action === 'syncWithDesktopApp') {
    DesktopAppConnection.syncLicenseStatus().then(sendResponse);
    return true; // Async response
  }

  // Get available themes
  else if (request.action === 'getAvailableThemes') {
    sendResponse({
      themes: StreamShieldPremium.getAvailableThemes(),
      allThemes: [...StreamShieldPremium.FREE_THEMES, ...StreamShieldPremium.PREMIUM_THEMES],
      isPremium: StreamShieldPremium.isPremium
    });
    return true;
  }

  // Check if theme is available
  else if (request.action === 'hasTheme') {
    const hasIt = StreamShieldPremium.hasTheme(request.theme);
    sendResponse({
      available: hasIt,
      requiresPremium: !hasIt
    });
    return true;
  }

  return true; // Keep channel open for async response
});

// ============================================================================
// Context Menus (Extended for v4.0)
// ============================================================================

function createContextMenus() {
  // Remove existing menus first
  chrome.contextMenus.removeAll(() => {
    // Whitelist this domain
    chrome.contextMenus.create({
      id: "streamshield-whitelist",
      title: "StreamShield: Whitelist this domain",
      contexts: ["page"]
    });

    // Block this element - appears on right-click of any element (v4.0 Premium)
    chrome.contextMenus.create({
      id: "streamshield-block-element",
      title: "StreamShield: Block this element",
      contexts: ["page", "image", "video", "audio", "link", "frame"]
    });

    // Launch element picker (v4.0)
    chrome.contextMenus.create({
      id: "streamshield-element-picker",
      title: "StreamShield: Pick an element to block",
      contexts: ["page"]
    });

    // Report broken page (v4.0)
    chrome.contextMenus.create({
      id: "streamshield-report-broken",
      title: "StreamShield: Report broken page",
      contexts: ["page"]
    });

    // Separator
    chrome.contextMenus.create({
      id: "streamshield-separator",
      type: "separator",
      contexts: ["page"]
    });

    // Premium status indicator
    chrome.contextMenus.create({
      id: "streamshield-premium-status",
      title: StreamShieldPremium.isPremium ? "StreamShield Premium: Active" : "StreamShield: Upgrade to Premium",
      contexts: ["page"]
    });
  });
}

chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === "streamshield-whitelist") {
    const url = new URL(tab.url);
    const domain = url.hostname;
    whitelist.add(domain);
    const whitelistArray = Array.from(whitelist);
    // Save to both storages
    chrome.storage.local.set({ whitelist: whitelistArray });
    chrome.storage.sync.set({ whitelist: whitelistArray });
    // Sync to proxy
    syncWhitelistToProxy();

    // Reload page
    chrome.tabs.reload(tab.id);
  }
  else if (info.menuItemId === "streamshield-block-element") {
    // Block this element - premium feature
    if (!StreamShieldPremium.hasFeature('element_picker')) {
      // Show notification about premium requirement
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'Premium Feature',
        message: 'Element blocking requires StreamShield Premium. Click to upgrade!',
        buttons: [{ title: 'Upgrade Now' }]
      });
      return;
    }
    // Launch element picker to select and block element
    launchElementPicker(tab.id);
  }
  else if (info.menuItemId === "streamshield-element-picker") {
    // Check premium before launching
    if (!StreamShieldPremium.hasFeature('element_picker')) {
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'Premium Feature',
        message: 'Element Picker requires StreamShield Premium. Click to upgrade!',
        buttons: [{ title: 'Upgrade Now' }]
      });
      return;
    }
    launchElementPicker(tab.id);
  }
  else if (info.menuItemId === "streamshield-report-broken") {
    // Open report page or send report
    const url = new URL(tab.url);
    const reportUrl = `${CONFIG.dashboardUrl}/report?domain=${encodeURIComponent(url.hostname)}&url=${encodeURIComponent(tab.url)}`;
    chrome.tabs.create({ url: reportUrl });
  }
  else if (info.menuItemId === "streamshield-premium-status") {
    // Open premium page or options
    if (StreamShieldPremium.isPremium) {
      chrome.tabs.create({ url: chrome.runtime.getURL('options.html') });
    } else {
      chrome.tabs.create({ url: 'https://streamshield.io/premium' });
    }
  }
});

// Initialize context menus on load
createContextMenus();

// ============================================================================
// KEYBOARD SHORTCUTS
// ============================================================================

chrome.commands.onCommand.addListener(async (command) => {
  if (command === 'toggle-element-picker') {
    // Check premium before launching
    if (!StreamShieldPremium.hasFeature('element_picker')) {
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'Premium Feature',
        message: 'Element Picker requires StreamShield Premium. Upgrade to unlock!',
        buttons: [{ title: 'Upgrade Now' }]
      });
      return;
    }

    // Get active tab and launch element picker
    try {
      const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (activeTab && activeTab.id) {
        // Check if we can inject into this tab
        if (activeTab.url && (activeTab.url.startsWith('http://') || activeTab.url.startsWith('https://'))) {
          launchElementPicker(activeTab.id);
        } else {
          console.log('Cannot inject element picker into this page:', activeTab.url);
        }
      }
    } catch (error) {
      console.error('Error launching element picker via keyboard shortcut:', error);
    }
  }
});

console.log('Keyboard shortcuts initialized');

// ============================================================================
// IMMEDIATE INITIALIZATION
// ============================================================================

// Initialize premium features immediately
StreamShieldPremium.init().then(() => {
  // Update context menus with premium status
  createContextMenus();
  // Update badge
  updateBadge();
});

// Load custom rules on startup
loadCustomRules();

// Try to connect to desktop app on startup
DesktopAppConnection.connect().then(result => {
  if (result.success) {
    console.log('Connected to StreamShield Desktop App on startup');
    DesktopAppConnection.syncLicenseStatus();
  }
});

// Handle notification button clicks
chrome.notifications.onButtonClicked.addListener((notificationId, buttonIndex) => {
  if (buttonIndex === 0) {
    // Upgrade button clicked
    chrome.tabs.create({ url: 'https://streamshield.io/premium' });
  }
  chrome.notifications.clear(notificationId);
});

console.log('StreamShield Background Script v4.0.0 Loaded with Premium Features!');
