# ZoeChip Popup Blocking Fix

## Overview
This document describes the enhanced popup blocking specifically designed for zoechip.com and similar streaming sites.

## Changes Made

### 1. New File: `zoechip-popup-blocker.js`
A dedicated, ultra-aggressive popup blocker that runs ONLY on streaming sites including:
- zoechip.com, zoechip.cc, zoechip.gg
- fmovies, gomovies, lookmovie
- 123movies, soap2day, putlocker
- movie4k, flixmomo, rabbitstream
- vidstream, dokicloud, streamtape, mixdrop

### 2. Updated: `manifest.json`
Added a second content script entry that loads `zoechip-popup-blocker.js` specifically for streaming sites.

## Features

### Aggressive Popup Blocking
- **Complete window.open() blocking** - NO exceptions for streaming sites
- **Click hijacking prevention** - Blocks ad links clicked by users
- **Redirect loop detection** - Prevents malicious redirect chains
- **Location.replace/assign blocking** - Stops forced redirects

### Timer-Based Popup Prevention
- Blocks popups from `setTimeout()` and `setInterval()`
- Prevents delayed popups that try to circumvent click detection

### Overlay Removal
- Removes fake video player overlays
- Detects and removes suspicious fixed/absolute positioned elements
- Removes elements with high z-index (common for ads)
- Preserves actual video players and embeds
- Runs every 500ms for aggressive removal

### Ad URL Pattern Detection
Blocks clicks to known ad networks:
- popads, popunder, adsterra, exoclick
- juicyads, propellerads, admaven, clickadu
- trafficjunky, bidvertiser, onclickads
- highperformanceformat, streamads.net
- And 20+ more ad networks

### Attribute Removal
- Strips `onclick` handlers that open popups
- Prevents document-level event handler injection
- Removes ad container elements by ID/class

### Iframe Blocking
- Monitors DOM for new iframe creation
- Automatically removes iframes with ad URLs or ad-related IDs/classes

## How It Works

### 1. Site Detection
The script first checks if you're on a streaming site:
```javascript
const isStreamingSite = streamingSites.some(site => hostname.includes(site));
```

If not a streaming site, it exits early and doesn't interfere.

### 2. User Click Tracking
Tracks genuine user clicks vs. programmatic redirects:
```javascript
document.addEventListener('click', function(e) {
  isUserClick = true;
  clickTime = Date.now();
  setTimeout(() => { isUserClick = false; }, 100);
});
```

### 3. Overlay Detection
Identifies suspicious overlays by:
- Position (fixed/absolute)
- Z-index > 99
- Size (large overlays covering 30%+ of viewport)
- Keywords in class/id (ad, popup, banner, modal, overlay)

### 4. Smart Preservation
Does NOT remove:
- Video elements (`<video>`)
- Iframes with "embed" or "player" in URL
- Small navigation elements (< 300x100px)

## Testing

To test if the popup blocker is working:

1. Visit zoechip.com or any supported streaming site
2. Open browser console (F12)
3. Look for messages:
   - `🛡️ StreamShield: Aggressive Popup/Redirect Blocker Active`
   - `✅ Free streaming popup/redirect blocker loaded`

4. Try clicking around the page - you should see:
   - `🚫 BLOCKED popup:` when popups are blocked
   - `🚫 BLOCKED ad link click:` when ad links are clicked
   - `🗑️ Removed suspicious overlay:` when overlays are removed

5. Check stats every 30 seconds:
   - `📊 StreamShield: Blocked X popups/redirects`

## Compatibility

### Works With:
- ✅ StreamShield standalone extension
- ✅ StreamShield proxy (both can run simultaneously)
- ✅ Chrome, Edge, Brave, and Chromium-based browsers
- ✅ Manifest V3

### Does NOT Break:
- ✅ Actual video playback
- ✅ Embedded video players
- ✅ Site navigation
- ✅ Legitimate user interactions

## Known Limitations

1. **Some sites may still show popups on first click** if they use very sophisticated techniques
2. **Cannot block popups opened before the script loads** - that's why `run_at: "document_start"` is critical
3. **Some legitimate site features** might be blocked if they use similar patterns to ads (rare)

## Troubleshooting

### If popups still appear:

1. **Check that extension is loaded**:
   - Open browser console
   - Look for "StreamShield" messages

2. **Verify the script is running**:
   - Console should show: `🛡️ StreamShield: Aggressive Popup/Redirect Blocker Active`

3. **Check for script errors**:
   - Look for red errors in console
   - Some sites use advanced CSP (Content Security Policy) that might block our scripts

4. **Try disabling other extensions**:
   - Some ad blockers might conflict
   - Test with only StreamShield enabled

5. **Clear browser cache**:
   - Some sites cache old scripts
   - Hard refresh: Ctrl+Shift+R

### If video doesn't play:

1. **Check if video player was mistakenly removed**:
   - Look for messages: `🗑️ Removed suspicious overlay`
   - Reload the page

2. **Whitelist the site** (if needed):
   - Add site to StreamShield whitelist via popup
   - This will disable aggressive blocking

## Integration with Proxy

This extension works seamlessly with the StreamShield proxy:

1. **Extension handles**: Client-side popup blocking, DOM manipulation
2. **Proxy handles**: Network-level blocking, HTML injection, script modification

Both can run simultaneously for maximum protection!

### Proxy Script Injection
The proxy (`script_blocker.py`) injects similar popup blocking code for zoechip:
```python
if any(domain in domain_lower for domain in [
    'zoechip.gg', 'zoechip.com',
    ...
]):
    return HTMLInjector.inject_script_tag(html, self.FREE_STREAMING_POPUP_BLOCKER)
```

This provides **defense in depth** - even if one layer fails, the other catches it.

## Future Enhancements

Potential improvements:
- [ ] Machine learning-based popup detection
- [ ] Site-specific rules database
- [ ] User reporting system for new popup patterns
- [ ] Automatic pattern learning from blocked popups
- [ ] Integration with SponsorBlock API for ad segment detection

## Credits

Based on the aggressive popup blocker from StreamShield proxy (`script_blocker.py`), enhanced for browser extension use.

## License

Part of StreamShield - MIT License
